#!/usr/bin/env python3
"""
Standalone script to replot judge scores from LLMCompare evaluation results.
All plotting logic is reimplemented here for easy modification and testing.

Usage:
    python scripts/replot_llmcompare.py path/to/results.json [--output-dir custom/output/dir]
"""

import argparse
import json
import sys
from collections import defaultdict
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional

import matplotlib.pyplot as plt
import numpy as np
import yaml


class StandaloneLLMComparePlotter:
    """Standalone plotter for LLMCompare evaluation results."""

    def __init__(self, results_path: Path):
        """Initialize plotter with results file path."""
        self.results_path = results_path
        with open(results_path, "r") as f:
            self.data = json.load(f)

        # Build mapping of questions to their source YAML files
        self.question_to_yaml = self._build_question_yaml_mapping()

        # Build judge metadata cache
        self.judge_metadata = self._build_judge_metadata()

        # Set up plot style
        plt.style.use("seaborn-v0_8-whitegrid")
        plt.rcParams["font.family"] = "sans-serif"
        plt.rcParams["font.size"] = 10
        plt.rcParams["figure.figsize"] = (12, 6)

    def _build_question_yaml_mapping(self) -> Dict[str, str]:
        """Build mapping of question IDs to their source YAML files."""
        question_to_yaml = {}

        # Get question files from config
        question_files = self.data.get("config", {}).get("question_files", [])

        # Load each YAML file to find which questions it contains
        for yaml_file_path in question_files:
            yaml_path = Path(yaml_file_path)
            if yaml_path.exists():
                try:
                    with open(yaml_path, "r") as f:
                        yaml_content = yaml.safe_load(f)

                    # Map each question to this YAML file
                    for question_id in yaml_content.keys():
                        # Get just the filename without path for cleaner subdirs
                        yaml_name = yaml_path.stem  # e.g., "helpfulness_eval"
                        question_to_yaml[question_id] = yaml_name
                except Exception as e:
                    print(f"Warning: Could not load {yaml_path}: {e}")

        return question_to_yaml

    def _build_judge_metadata(self) -> Dict[str, Dict]:
        """Build mapping of judge IDs to their metadata including ranges."""
        judge_metadata = {}

        # Get question files from config
        question_files = self.data.get("config", {}).get("question_files", [])

        # Load each YAML file to find judge definitions
        for yaml_file_path in question_files:
            yaml_path = Path(yaml_file_path)
            if yaml_path.exists():
                try:
                    with open(yaml_path, "r") as f:
                        yaml_content = yaml.safe_load(f)

                    # Extract judge metadata
                    for item_id, item_data in yaml_content.items():
                        if isinstance(item_data, dict) and "judge" in item_data.get(
                            "type", ""
                        ):
                            # Store judge metadata including range
                            judge_metadata[item_id] = {
                                "type": item_data.get("type"),
                                "min_score": item_data.get("min_score", 0),
                                "max_score": item_data.get("max_score", 100),
                                "scale": item_data.get(
                                    "scale", None
                                ),  # e.g., "0-10", "1-5"
                            }
                except Exception as e:
                    print(
                        f"Warning: Could not load judge metadata from {yaml_path}: {e}"
                    )

        return judge_metadata

    def create_plots(
        self, output_dir: Optional[Path] = None, timestamp: Optional[str] = None
    ) -> List[Path]:
        """Create all plots for the evaluation results."""
        if output_dir is None:
            output_dir = Path("images/llmcompare_eval")

        # Use provided timestamp or extract from results data
        if timestamp is None:
            timestamp = self.data.get(
                "timestamp", datetime.now().strftime("%Y%m%d_%H%M%S")
            )

        plot_dir = output_dir / f"{timestamp}_llmcompare"
        plot_dir.mkdir(parents=True, exist_ok=True)

        plot_files = []

        # Create rating task plots
        if self._has_rating_tasks():
            rating_plots = self._create_rating_plots(plot_dir)
            plot_files.extend(rating_plots)

        # Create judge score plots
        if self._has_judge_scores():
            judge_plots = self._create_judge_score_plots(plot_dir)
            plot_files.extend(judge_plots)

        # Create response length plots
        length_plots = self._create_response_length_plot(plot_dir)
        plot_files.extend(length_plots)

        return plot_files

    def _has_judge_scores(self) -> bool:
        """Check if results contain judge scores."""
        for result in self.data.get("results", []):
            if result.get("judge_scores"):
                return True
        return False

    def _organize_data_by_question(self) -> Dict:
        """Organize results by question and model."""
        organized = defaultdict(lambda: defaultdict(list))

        for result in self.data.get("results", []):
            question_id = result["question_id"]
            model_ref = result["model_ref"]
            organized[question_id][model_ref].append(result)

        return organized

    def _extract_judge_scores(self, results: List[Dict], judge_id: str) -> List[float]:
        """Extract numeric scores for a specific judge from results, excluding non-numeric responses."""
        scores = []
        for result in results:
            judge_data = result.get("judge_scores", {}).get(judge_id, {})
            # Only include numeric scores, skip non-numeric responses
            if "score" in judge_data and judge_data["score"] != 0:
                scores.append(judge_data["score"])
            # Note: We intentionally skip entries with 'non_numeric_response'
        return scores

    def _count_non_numeric_responses(self, results: List[Dict], judge_id: str) -> int:
        """Count non-numeric responses for a specific judge."""
        count = 0
        for result in results:
            judge_data = result.get("judge_scores", {}).get(judge_id, {})
            if "non_numeric_response" in judge_data:
                count += 1
        return count

    def _extract_answer_from_tags(self, text: str) -> Optional[str]:
        """Extract answer from <answer></answer> tags."""
        import re

        match = re.search(r"<answer>(.*?)</answer>", text, re.IGNORECASE | re.DOTALL)
        if match:
            return match.group(1).strip()
        return None

    def _has_rating_tasks(self) -> bool:
        """Check if results contain rating type tasks."""
        for result in self.data.get("results", []):
            if result.get("metadata", {}).get("question_type") == "rating":
                return True
        return False

    def _create_rating_plots(self, plot_dir: Path) -> List[Path]:
        """Create plots for rating type tasks organized by YAML file."""
        organized_data = self._organize_data_by_question()
        plot_files = []

        # Group questions by their source YAML file, filtering for rating type
        yaml_to_questions = defaultdict(list)
        questions_with_compliance = defaultdict(bool)

        for question_id in organized_data.keys():
            # Check if this is a rating type question
            is_rating = False
            has_compliance_tracking = False

            for model_results in organized_data[question_id].values():
                if (
                    model_results
                    and model_results[0].get("metadata", {}).get("question_type")
                    == "rating"
                ):
                    is_rating = True
                    # Check if any result has compliance tracking
                    if any(
                        r.get("metadata", {}).get("is_format_compliant") is not None
                        for r in model_results
                    ):
                        has_compliance_tracking = True
                    break

            if is_rating:
                yaml_name = self.question_to_yaml.get(question_id, "unknown")
                yaml_to_questions[yaml_name].append(question_id)
                questions_with_compliance[question_id] = has_compliance_tracking

        # Create plots for each YAML file
        for yaml_name, questions in yaml_to_questions.items():
            if not questions:
                continue

            # Create subdirectory for this YAML
            yaml_dir = plot_dir / yaml_name
            yaml_dir.mkdir(parents=True, exist_ok=True)

            # Check if any questions in this YAML have compliance tracking
            has_any_compliance = any(questions_with_compliance[q] for q in questions)

            # Create figure with subplots if compliance tracking is enabled
            if has_any_compliance:
                fig, (ax1, ax2) = plt.subplots(
                    2,
                    1,
                    figsize=(max(12, len(questions) * 1.5), 10),
                    height_ratios=[2, 1],
                )
            else:
                fig, ax1 = plt.subplots(figsize=(max(12, len(questions) * 1.5), 6))

            # Prepare data for plotting
            questions_to_plot = sorted(questions)
            models = sorted(
                set(
                    model
                    for q in questions_to_plot
                    for model in organized_data[q].keys()
                )
            )

            # Assign colors to models
            colors = plt.cm.tab10(np.linspace(0, 1, len(models)))
            model_colors = {model: colors[i] for i, model in enumerate(models)}

            # Plot each question
            x_positions = []
            x_labels = []

            for q_idx, question in enumerate(questions_to_plot):
                x_base = q_idx * (len(models) + 1)
                x_center = x_base + len(models) / 2 - 0.5
                x_positions.append(x_center)
                x_labels.append(question.replace("_", " ").title())

                for m_idx, model in enumerate(models):
                    if model in organized_data[question]:
                        results = organized_data[question][model]
                        ratings = []

                        # Extract ratings from metadata
                        for result in results:
                            extracted_rating = result.get("metadata", {}).get(
                                "extracted_rating"
                            )
                            if extracted_rating is not None:
                                ratings.append(extracted_rating)

                        if ratings:
                            # Center models around the label position
                            x_offset = (m_idx - (len(models) - 1) / 2) * 0.3
                            x_pos = x_center + x_offset

                            if len(ratings) == 1:
                                # Single sample - just plot the point
                                ax1.scatter(
                                    x_pos,
                                    ratings[0],
                                    color=model_colors[model],
                                    s=50,
                                    zorder=3,
                                )
                            else:
                                # Multiple samples - plot with error bars
                                mean_rating = np.mean(ratings)
                                std_rating = np.std(ratings)
                                ax1.errorbar(
                                    x_pos,
                                    mean_rating,
                                    yerr=std_rating,
                                    fmt="o",
                                    color=model_colors[model],
                                    capsize=5,
                                    capthick=2,
                                    zorder=3,
                                )

            # Customize main plot
            ax1.set_xticks(x_positions)
            if has_any_compliance:
                ax1.set_xticklabels(
                    []
                )  # Hide x labels for top plot when we have subplots
            else:
                ax1.set_xticklabels(x_labels, rotation=45, ha="right")
            ax1.set_ylabel("Rating")
            ax1.grid(True, alpha=0.3)

            # Add legend
            legend_elements = [
                plt.Line2D(
                    [0],
                    [0],
                    marker="o",
                    color="w",
                    markerfacecolor=model_colors[model],
                    markersize=8,
                    label=model,
                )
                for model in models
            ]
            ax1.legend(
                handles=legend_elements,
                loc="upper right",
                bbox_to_anchor=(1.0, 1.0),
                ncol=len(models),
            )

            # Count total samples across all questions and models
            total_samples = 0
            for question in questions_to_plot:
                for model in models:
                    if model in organized_data[question]:
                        total_samples += len(organized_data[question][model])

            # Add title with sample count to main plot
            ax1.set_title(
                f"Rating Task Results - {yaml_name.replace('_', ' ').title()}\n(n = {total_samples} total samples)"
            )

            # Create compliance subplot if enabled
            if has_any_compliance:
                # Calculate compliance percentages
                for q_idx, question in enumerate(questions_to_plot):
                    # Only process questions with compliance tracking
                    if not questions_with_compliance[question]:
                        continue

                    x_center = x_positions[q_idx]

                    for m_idx, model in enumerate(models):
                        if model in organized_data[question]:
                            results = organized_data[question][model]

                            # Count compliant responses
                            compliant_count = 0
                            total_count = 0

                            for result in results:
                                if (
                                    result.get("metadata", {}).get(
                                        "is_format_compliant"
                                    )
                                    is not None
                                ):
                                    total_count += 1
                                    if result.get("metadata", {}).get(
                                        "is_format_compliant"
                                    ):
                                        compliant_count += 1

                            if total_count > 0:
                                compliance_percentage = (
                                    compliant_count / total_count
                                ) * 100

                                # Center models around the label position
                                x_offset = (m_idx - (len(models) - 1) / 2) * 0.3
                                x_pos = x_center + x_offset

                                # Plot compliance percentage as a bar
                                ax2.bar(
                                    x_pos,
                                    compliance_percentage,
                                    width=0.25,
                                    color=model_colors[model],
                                    alpha=0.7,
                                )

                # Customize compliance subplot
                ax2.set_xticks(x_positions)
                ax2.set_xticklabels(x_labels, rotation=45, ha="right")
                ax2.set_ylabel("Format Compliance %")
                ax2.set_ylim(0, 105)
                ax2.grid(True, alpha=0.3, axis="y")

                # Add a note about compliance tracking
                questions_with_tracking = sum(
                    1 for q in questions_to_plot if questions_with_compliance[q]
                )
                ax2.set_xlabel(
                    f"Questions ({questions_with_tracking} of {len(questions_to_plot)} have compliance tracking enabled)"
                )

            plt.tight_layout()

            # Save plot in YAML subdirectory
            plot_file = yaml_dir / "rating_tasks.png"
            plt.savefig(plot_file, dpi=150, bbox_inches="tight")
            plt.close()
            plot_files.append(plot_file)

            print(f"Created plot: {plot_file}")

        return plot_files

    def _create_judge_score_plots(self, plot_dir: Path) -> List[Path]:
        """Create plots for judge scores organized by YAML file."""
        organized_data = self._organize_data_by_question()
        plot_files = []

        # Group questions by their source YAML file
        yaml_to_questions = defaultdict(list)
        for question_id in organized_data.keys():
            yaml_name = self.question_to_yaml.get(question_id, "unknown")
            yaml_to_questions[yaml_name].append(question_id)

        # Create plots for each YAML file
        for yaml_name, questions in yaml_to_questions.items():
            # Create subdirectory for this YAML
            yaml_dir = plot_dir / yaml_name
            yaml_dir.mkdir(parents=True, exist_ok=True)

            # Get judges that are actually used by questions in this YAML
            yaml_judges = set()
            for result in self.data.get("results", []):
                if result["question_id"] in questions and "judge_scores" in result:
                    yaml_judges.update(result["judge_scores"].keys())

            # Create a plot for each judge used in this YAML
            for judge_id in sorted(yaml_judges):
                # Check if this is an answer judge by looking at the YAML data
                is_answer_judge = False
                for yaml_file_path in self.data.get("config", {}).get(
                    "question_files", []
                ):
                    yaml_path = Path(yaml_file_path)
                    if yaml_path.exists():
                        try:
                            with open(yaml_path, "r") as f:
                                yaml_content = yaml.safe_load(f)
                            # Check if this judge is defined as an answer_judge
                            if judge_id in yaml_content:
                                judge_type = yaml_content[judge_id].get("type", "")
                                if "answer_judge" in judge_type:
                                    is_answer_judge = True
                                    break
                        except Exception:
                            pass

                # Filter questions to only include those where this judge was actually used
                questions_with_judge = []
                for question in questions:
                    for model in organized_data[question]:
                        for result in organized_data[question][model]:
                            if judge_id in result.get("judge_scores", {}):
                                questions_with_judge.append(question)
                                break
                        if question in questions_with_judge:
                            break

                if not questions_with_judge:
                    continue

                if is_answer_judge:
                    # Create answer distribution plot
                    plot_file = self._create_answer_judge_plot(
                        yaml_dir, judge_id, questions_with_judge, organized_data
                    )
                    if plot_file:
                        plot_files.append(plot_file)
                        print(f"Created plot: {plot_file}")
                else:
                    # Create regular score plot
                    fig, ax = plt.subplots(
                        figsize=(max(12, len(questions_with_judge) * 1.5), 6)
                    )

                    # Prepare data for plotting
                    questions_to_plot = sorted(questions_with_judge)
                    models = sorted(
                        set(
                            model
                            for q in questions_to_plot
                            for model in organized_data[q].keys()
                        )
                    )

                    # Assign colors to models
                    colors = plt.cm.tab10(np.linspace(0, 1, len(models)))
                    model_colors = {model: colors[i] for i, model in enumerate(models)}

                    # Plot each question
                    x_positions = []
                    x_labels = []

                    for q_idx, question in enumerate(questions_to_plot):
                        x_base = q_idx * (len(models) + 1)
                        x_center = x_base + len(models) / 2 - 0.5
                        x_positions.append(x_center)
                        x_labels.append(question.replace("_", " ").title())

                        for m_idx, model in enumerate(models):
                            if model in organized_data[question]:
                                results = organized_data[question][model]
                                scores = self._extract_judge_scores(results, judge_id)

                                if scores:
                                    # Center models around the label position
                                    x_offset = (m_idx - (len(models) - 1) / 2) * 0.3
                                    x_pos = x_center + x_offset

                                    if len(scores) == 1:
                                        # Single sample - just plot the point
                                        ax.scatter(
                                            x_pos,
                                            scores[0],
                                            color=model_colors[model],
                                            s=50,
                                            zorder=3,
                                        )
                                    else:
                                        # Multiple samples - plot with error bars
                                        mean_score = np.mean(scores)
                                        std_score = np.std(scores)
                                        ax.errorbar(
                                            x_pos,
                                            mean_score,
                                            yerr=std_score,
                                            fmt="o",
                                            color=model_colors[model],
                                            capsize=5,
                                            capthick=2,
                                            zorder=3,
                                        )

                    # Customize plot
                    ax.set_xticks(x_positions)
                    ax.set_xticklabels(x_labels, rotation=45, ha="right")
                    ax.set_ylabel(f"{judge_id.replace('_', ' ').title()} Score")

                    # Get judge range from metadata
                    judge_meta = self.judge_metadata.get(judge_id, {})
                    min_score = judge_meta.get("min_score", 0)
                    max_score = judge_meta.get("max_score", 100)

                    # Set y-axis limits with some padding
                    y_padding = (max_score - min_score) * 0.05
                    ax.set_ylim(min_score - y_padding, max_score + y_padding)
                    ax.grid(True, alpha=0.3)

                    # Add legend
                    legend_elements = [
                        plt.Line2D(
                            [0],
                            [0],
                            marker="o",
                            color="w",
                            markerfacecolor=model_colors[model],
                            markersize=8,
                            label=model,
                        )
                        for model in models
                    ]
                    ax.legend(
                        handles=legend_elements,
                        loc="upper right",
                        bbox_to_anchor=(1.0, 1.0),
                        ncol=len(models),
                    )

                    # Count total samples and non-numeric responses across all questions and models
                    total_samples = 0
                    non_numeric_count = 0
                    for question in questions_to_plot:
                        for model in models:
                            if model in organized_data[question]:
                                results = organized_data[question][model]
                                total_samples += len(results)
                                non_numeric_count += self._count_non_numeric_responses(
                                    results, judge_id
                                )

                    # Add title with sample count
                    judge_title = judge_id.replace("_", " ").title()
                    if non_numeric_count > 0:
                        ax.set_title(
                            f"{judge_title} Scores - {yaml_name.replace('_', ' ').title()}\n(n = {total_samples} total samples, {non_numeric_count} non-numeric responses excluded)"
                        )
                    else:
                        ax.set_title(
                            f"{judge_title} Scores - {yaml_name.replace('_', ' ').title()}\n(n = {total_samples} total samples)"
                        )

                    plt.tight_layout()

                    # Save plot in YAML subdirectory
                    plot_file = yaml_dir / f"judge_scores_{judge_id}.png"
                    plt.savefig(plot_file, dpi=150, bbox_inches="tight")
                    plt.close()
                    plot_files.append(plot_file)

                    print(f"Created plot: {plot_file}")

        return plot_files

    def _create_answer_judge_plot(
        self, yaml_dir: Path, judge_id: str, questions: List[str], organized_data: Dict
    ) -> Optional[Path]:
        """Create plot for answer judges showing answers as points with numeric mapping."""
        fig, ax = plt.subplots(figsize=(max(12, len(questions) * 1.5), 6))

        # Prepare data
        questions = sorted(questions)
        models = sorted(
            set(model for q in questions for model in organized_data[q].keys())
        )

        # Collect all answers for each question and model
        answer_data = defaultdict(lambda: defaultdict(list))

        for question in questions:
            for model in models:
                if model in organized_data[question]:
                    results = organized_data[question][model]
                    for result in results:
                        judge_data = result.get("judge_scores", {}).get(judge_id, {})
                        # Look for response in either 'raw_response' or 'evaluation' field
                        response_text = judge_data.get(
                            "raw_response", ""
                        ) or judge_data.get("evaluation", "")
                        answer = self._extract_answer_from_tags(response_text)
                        if answer:
                            answer_data[question][model].append(answer.lower())

        # Get all unique answers and create numeric mapping
        all_answers = set()
        for q_data in answer_data.values():
            for answers in q_data.values():
                all_answers.update(answers)
        all_answers = sorted(all_answers)

        if not all_answers:
            return None

        # Get judge range from metadata
        judge_meta = self.judge_metadata.get(judge_id, {})
        min_score = judge_meta.get("min_score", 0)
        max_score = judge_meta.get("max_score", 100)

        # Create numeric mapping for answers
        # For appropriate/inappropriate, use max/min for clear interpretation
        if set(all_answers) == {"appropriate", "inappropriate"}:
            answer_to_score = {"appropriate": max_score, "inappropriate": min_score}
        elif len(all_answers) == 1:
            # If there's only one answer type, put it at midpoint
            answer_to_score = {all_answers[0]: (min_score + max_score) / 2}
        else:
            # For other categorical answers, space them evenly
            score_range = max_score - min_score
            answer_to_score = {
                answer: min_score + i * (score_range / (len(all_answers) - 1))
                for i, answer in enumerate(all_answers)
            }

        # Assign colors to models
        colors = plt.cm.tab10(np.linspace(0, 1, len(models)))
        model_colors = {model: colors[i] for i, model in enumerate(models)}

        # Plot the data
        x_positions = []
        x_labels = []

        for q_idx, question in enumerate(questions):
            x_base = q_idx * (len(models) + 1)
            x_center = x_base + len(models) / 2 - 0.5
            x_positions.append(x_center)
            x_labels.append(question.replace("_", " ").title())

            for m_idx, model in enumerate(models):
                if model in answer_data[question] and answer_data[question][model]:
                    # Center models around the label position
                    x_offset = (m_idx - (len(models) - 1) / 2) * 0.3
                    x_pos = x_center + x_offset

                    # Convert answers to scores
                    scores = [
                        answer_to_score[answer]
                        for answer in answer_data[question][model]
                    ]

                    if len(scores) == 1:
                        # Single sample - just plot the point
                        ax.scatter(
                            x_pos, scores[0], color=model_colors[model], s=50, zorder=3
                        )
                    else:
                        # Multiple samples - plot with error bars
                        mean_score = np.mean(scores)
                        std_score = np.std(scores)
                        ax.errorbar(
                            x_pos,
                            mean_score,
                            yerr=std_score,
                            fmt="o",
                            color=model_colors[model],
                            capsize=5,
                            capthick=2,
                            zorder=3,
                        )

        # Customize plot
        ax.set_xticks(x_positions)
        ax.set_xticklabels(x_labels, rotation=45, ha="right")

        # Get judge range from metadata
        judge_meta = self.judge_metadata.get(judge_id, {})
        min_score = judge_meta.get("min_score", 0)
        max_score = judge_meta.get("max_score", 100)

        # Set y-axis labels based on answer types
        y_padding = (max_score - min_score) * 0.05
        ax.set_ylim(min_score - y_padding, max_score + y_padding)

        if set(all_answers) == {"appropriate", "inappropriate"}:
            ax.set_ylabel("Judge Response")
            ax.set_yticks([min_score, max_score])
            ax.set_yticklabels(["Inappropriate", "Appropriate"])
        else:
            ax.set_ylabel("Judge Response")
            ax.set_yticks([answer_to_score[answer] for answer in all_answers])
            ax.set_yticklabels([answer.title() for answer in all_answers])

        ax.grid(True, alpha=0.3)

        # Add legend for models
        legend_elements = [
            plt.Line2D(
                [0],
                [0],
                marker="o",
                color="w",
                markerfacecolor=model_colors[model],
                markersize=8,
                label=model,
            )
            for model in models
        ]
        ax.legend(
            handles=legend_elements,
            loc="upper right",
            bbox_to_anchor=(1.0, 1.0),
            ncol=len(models),
        )

        # Count total samples
        total_samples = sum(
            len(answers)
            for q_data in answer_data.values()
            for answers in q_data.values()
        )

        # Add title with sample count
        judge_title = judge_id.replace("_", " ").title()
        ax.set_title(
            f"{judge_title} - {yaml_dir.name.replace('_', ' ').title()}\n(n = {total_samples} total samples)"
        )

        plt.tight_layout()

        # Save plot with consistent naming
        plot_file = yaml_dir / f"judge_scores_{judge_id}.png"
        plt.savefig(plot_file, dpi=150, bbox_inches="tight")
        plt.close()

        return plot_file

    def _create_response_length_plot(self, plot_dir: Path) -> List[Path]:
        """Create plot showing response lengths across questions and models, organized by YAML."""
        organized_data = self._organize_data_by_question()
        plot_files = []

        # Group questions by their source YAML file
        yaml_to_questions = defaultdict(list)
        for question_id in organized_data.keys():
            yaml_name = self.question_to_yaml.get(question_id, "unknown")
            yaml_to_questions[yaml_name].append(question_id)

        # Create plots for each YAML file
        for yaml_name, questions in yaml_to_questions.items():
            # Create subdirectory for this YAML
            yaml_dir = plot_dir / yaml_name
            yaml_dir.mkdir(parents=True, exist_ok=True)

            fig, ax = plt.subplots(figsize=(max(12, len(questions) * 1.5), 6))

            questions = sorted(questions)
            models = sorted(
                set(model for q in questions for model in organized_data[q].keys())
            )

            # Assign colors to models
            colors = plt.cm.tab10(np.linspace(0, 1, len(models)))
            model_colors = {model: colors[i] for i, model in enumerate(models)}

            # Plot each question
            x_positions = []
            x_labels = []

            for q_idx, question in enumerate(questions):
                x_base = q_idx * (len(models) + 1)
                x_center = x_base + len(models) / 2 - 0.5
                x_positions.append(x_center)
                x_labels.append(question.replace("_", " ").title())

                for m_idx, model in enumerate(models):
                    if model in organized_data[question]:
                        results = organized_data[question][model]
                        lengths = [len(r["response"]) for r in results]

                        if lengths:
                            # Center models around the label position
                            x_offset = (m_idx - (len(models) - 1) / 2) * 0.3
                            x_pos = x_center + x_offset

                            if len(lengths) == 1:
                                ax.scatter(
                                    x_pos,
                                    lengths[0],
                                    color=model_colors[model],
                                    s=50,
                                    zorder=3,
                                )
                            else:
                                mean_length = np.mean(lengths)
                                std_length = np.std(lengths)
                                ax.errorbar(
                                    x_pos,
                                    mean_length,
                                    yerr=std_length,
                                    fmt="o",
                                    color=model_colors[model],
                                    capsize=5,
                                    capthick=2,
                                    zorder=3,
                                )

            # Customize plot
            ax.set_xticks(x_positions)
            ax.set_xticklabels(x_labels, rotation=45, ha="right")
            ax.set_ylabel("Response Length (characters)")
            ax.grid(True, alpha=0.3)

            # Add legend
            legend_elements = [
                plt.Line2D(
                    [0],
                    [0],
                    marker="o",
                    color="w",
                    markerfacecolor=model_colors[model],
                    markersize=8,
                    label=model,
                )
                for model in models
            ]
            ax.legend(
                handles=legend_elements,
                loc="upper right",
                bbox_to_anchor=(1.0, 1.0),
                ncol=len(models),
            )

            # Count total samples
            total_samples = 0
            for question in questions:
                for model in models:
                    if model in organized_data[question]:
                        total_samples += len(organized_data[question][model])

            ax.set_title(
                f"Response Lengths - {yaml_name.replace('_', ' ').title()}\n(n = {total_samples} total samples)"
            )

            plt.tight_layout()

            # Save plot in YAML subdirectory
            plot_file = yaml_dir / "response_lengths.png"
            plt.savefig(plot_file, dpi=150, bbox_inches="tight")
            plt.close()
            plot_files.append(plot_file)

            print(f"Created plot: {plot_file}")

        return plot_files


def main():
    parser = argparse.ArgumentParser(
        description="Replot judge scores from LLMCompare evaluation results"
    )
    parser.add_argument(
        "results_file",
        type=Path,
        help="Path to the results JSON file from LLMCompare evaluation",
    )
    parser.add_argument(
        "--output-dir",
        type=Path,
        default=None,
        help="Output directory for plots (default: images/llmcompare_eval)",
    )
    parser.add_argument(
        "--timestamp",
        type=str,
        default=None,
        help="Override timestamp for output directory naming",
    )

    args = parser.parse_args()

    # Check if results file exists
    if not args.results_file.exists():
        print(f"Error: Results file not found: {args.results_file}")
        sys.exit(1)

    # Create plots
    print(f"Replotting results from: {args.results_file}")

    try:
        plotter = StandaloneLLMComparePlotter(args.results_file)
        plot_files = plotter.create_plots(
            output_dir=args.output_dir, timestamp=args.timestamp
        )

        if plot_files:
            print(f"\nSuccessfully created {len(plot_files)} plots:")
            for plot_file in plot_files:
                print(f"  - {plot_file}")
        else:
            print(
                "\nNo plots were created. Check if the results file contains judge scores."
            )

    except Exception as e:
        print(f"\nError creating plots: {e}")
        import traceback

        traceback.print_exc()
        sys.exit(1)


if __name__ == "__main__":
    main()
