#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
MCP客户端测试脚本
测试与MCP服务器的连接和基本功能
"""

import sys
import os

# 添加项目根目录到Python路径
script_dir = os.path.dirname(os.path.abspath(__file__))
project_root = os.path.abspath(os.path.join(script_dir, "../.."))
sys.path.append(project_root)

from task_tracker.mcp.client import MCPDatabaseClient, configure_mcp_client
import json
import tempfile

def test_mcp_client(server_url="http://localhost:8000"):
    """测试MCP客户端功能"""
    print(f"Testing MCP client connection to: {server_url}")
    
    try:
        # 配置客户端
        client = configure_mcp_client(server_url)
        
        # 测试1: 健康检查
        print("\n1. Testing health check...")
        response = client.session.get(f"{server_url}/health")
        if response.status_code == 200:
            print("✓ Health check passed")
        else:
            print("✗ Health check failed")
            return False
        
        # 测试2: 创建临时测试文档
        print("\n2. Creating test documents...")
        test_data = [
            {"context": "This is a test document about artificial intelligence."},
            {"context": "Machine learning is a subset of AI."},
            {"context": "Neural networks are powerful tools for pattern recognition."}
        ]
        
        with tempfile.NamedTemporaryFile(mode='w', suffix='.json', delete=False) as f:
            json.dump(test_data, f)
            test_file_path = f.name
        
        try:
            # 测试3: 加载文档
            print("\n3. Testing document loading...")
            documents = client.load_documents(test_file_path, "context")
            print(f"✓ Loaded {len(documents)} documents")
            
            # 测试4: 生成嵌入
            print("\n4. Testing embedding generation...")
            embeddings = client.generate_embeddings(
                ["test query"], 
                "/hub/huggingface/models/bert/bert-base-uncased"
            )
            print(f"✓ Generated embeddings shape: {embeddings.shape}")
            
            # 测试5: 构建索引
            print("\n5. Testing index building...")
            result = client.build_index(
                test_file_path,
                "/hub/huggingface/models/bert/bert-base-uncased",
                "test_index",
                batch_size=2
            )
            print(f"✓ Index built successfully: {result}")
            
            # 测试6: 搜索
            print("\n6. Testing search...")
            search_results = client.search(
                embeddings[0].tolist(),
                k=2,
                index_name="test_index"
            )
            print(f"✓ Search completed: {search_results}")
            
            # 测试7: 列出索引
            print("\n7. Testing index listing...")
            indexes = client.list_indexes()
            print(f"✓ Available indexes: {indexes}")
            
            print("\n🎉 All tests passed!")
            return True
            
        finally:
            # 清理临时文件
            os.unlink(test_file_path)
            
    except Exception as e:
        print(f"\n❌ Test failed: {e}")
        return False

def main():
    import argparse
    parser = argparse.ArgumentParser(description="Test MCP client functionality")
    parser.add_argument("--server", default="http://localhost:8000", 
                       help="MCP server URL")
    args = parser.parse_args()
    
    success = test_mcp_client(args.server)
    sys.exit(0 if success else 1)

if __name__ == "__main__":
    main()
