"""logging_utils
=================
Utility helpers to configure a **loguru** logger with optional colorized
console output that falls back to monochrome when running inside Azure ML.

Example:
    ```python
    from pathlib import Path

    from ml_utils import log, add_sink

    log.info("Hello world")
    add_sink(Path("debug.log"))
    ```
"""

import logging
import os
import sys
from pathlib import Path

import loguru
from loguru import logger


def _formatter(record) -> str:
    """Return a formatted log line for *record*.

    Args:
        record (dict[str, Any]): Record dictionary passed by **loguru** to the
            sink.

    Returns:
        str: A color‑ready or plain‑text formatted string consumed by *loguru*.
    """
    function_name = str(record["function"])
    top_module = record["name"].split(".", maxsplit=1)[0]
    if "<" in function_name or ">" in function_name:
        function_name = function_name.replace("<", r"\<").replace(">", r"\>")
    return (
        f"[<level>{record['level'].name:^5}</level>] "
        f"<cyan>{top_module:^15}</cyan> | "
        f"<cyan>{function_name:^20}</cyan> | "
        f"<level>{record['message']}</level>\n"
    )


class InterceptHandler(logging.Handler):
    """Forward standard‑library *logging* calls to **loguru**.

    External libraries that rely on the built‑in `logging.Logger` can
    therefore propagate their messages to the same sinks used by *loguru*.
    """

    def emit(self, record: logging.LogRecord) -> None:  # noqa: PLR6301
        """Forward *record* to **loguru**.

        Args:
            record (logging.LogRecord): The log record to forward.
        """
        level: int | str
        try:
            level = logger.level(record.levelname).name
        except ValueError:
            level = record.levelno

        frame = logging.currentframe()
        depth = 1
        while frame and frame.f_code.co_filename != record.pathname:
            frame = frame.f_back
            depth += 1

        logger.opt(depth=depth, exception=record.exc_info).log(level, record.getMessage())


def configure_logging() -> "loguru.Logger":
    """Configure and return the global **loguru** logger.

    Subsequent calls are no‑ops and simply return the already‑configured
    instance.

    Returns:
        loguru.Logger: The configured global logger.
    """

    # Remove default handler installed by **loguru**.
    logger.remove()

    use_color = os.environ.get("AZUREML_RUN_ID") is None

    logger.add(
        sink=sys.stderr,
        level="DEBUG",
        format=_formatter,
        colorize=use_color,
        backtrace=True,
        diagnose=True,
    )

    # Propagate standard-library logs to *loguru*.
    # logging.basicConfig(handlers=[InterceptHandler()], level=logging.CRITICAL)

    return logger


def add_sink(sink: Path | str) -> None:
    """Attach an additional **loguru** sink.

    Args:
        sink (Path | str): Path‑like object designating the sink. For example,
            ``Path("debug.log")``.
    """
    configure_logging()  # ensure logger is ready
    logger.add(
        sink,
        level="DEBUG",
        format=_formatter,
        colorize=False,
        backtrace=True,
        diagnose=True,
    )


# Configure the module-level logger instance that client code can import
log = configure_logging()
