from typing import Type

import os
import json

from src.methods.dfl_abstract import DFL
from src.methods.pfl import PFL
from src.methods.sfge import SFGE
from src.methods.gp_sfge import GaussianProcessSFGE
from src.methods.lodl import LODL
from src.methods.egl import EGL
from src.methods.spo import SPO
from src.methods.lancer import Lancer

from src.solvers.solver import Solver
from src.solvers.production_planning_solver import ProductionPlanningSolver
from src.solvers.toy_solver import ToySolver
from src.solvers.kp_values_solver import KnapsackValuesSolver
from src.solvers.kp_weights_solver import KnapsackWeightsSolver
from src.solvers.kp_capacity_solver import KnapsackCapacitySolver
from src.solvers.wsmc_solver import WeightedSetMultiCoverSolver

from src.readers.dataset_reader import DatasetReader
from src.readers.kp_values_reader import KnapsackValuesReader
from src.readers.kp_weights_reader import KnapsackWeightsReader
from src.readers.kp_capacity_reader import KnapsackCapacityReader
from src.readers.wsmc_reader import WeightedSetMultiCoverReader
from src.readers.toy_reader import ToyReader
from src.readers.production_planning_reader import ProductionPlanningReader

MODEL_CONFIGURATIONS_PATH = os.path.join("experiments", "model configurations")


class EnvironmentBuilder:

    METHODS: dict[str, Type[DFL]] = {
        "pfl": PFL,
        "sfge": SFGE,
        "gp_sfge": GaussianProcessSFGE,
        "lodl": LODL,
        "egl": EGL,
        "spo": SPO,
        "lancer": Lancer
    }

    PROBLEMS: dict[str, tuple[Type[Solver], Type[DatasetReader]], str] = {
        "kp_values": (KnapsackValuesSolver, KnapsackValuesReader, os.path.join("data", "kp values")),
        "kp_weights": (KnapsackWeightsSolver, KnapsackWeightsReader, os.path.join("data", "kp weights")),
        "kp_capacity": (KnapsackCapacitySolver, KnapsackCapacityReader, os.path.join("data", "kp capacity")),
        "wsmc": (WeightedSetMultiCoverSolver, WeightedSetMultiCoverReader, os.path.join("data", "wsmc")),
        "toy": (ToySolver, ToyReader, os.path.join("data", "toy")),
        "production_planning": (ProductionPlanningSolver, ProductionPlanningReader, os.path.join("data", "production planning"))
    }

    @staticmethod
    def parse_configuration(name: str, in_dim: int, out_dim: int) -> DFL:

        json_path = os.path.join(MODEL_CONFIGURATIONS_PATH, name + ".json")

        with open(json_path, 'r') as file:
            parameters = json.load(file)
            method = parameters["method"]

            assert method in EnvironmentBuilder.METHODS, "Invalid method: {}".format(method)

            return EnvironmentBuilder.METHODS[method].build_from_config(parameters, in_dim, out_dim)

    @staticmethod
    def parse_problem(problem: str, dataset_name: str) -> tuple[Solver, DatasetReader]:

        assert problem in EnvironmentBuilder.PROBLEMS, "Invalid problem: {}".format(problem)

        solver_cls, dataset_reader_cls, dataset_path = EnvironmentBuilder.PROBLEMS[problem]

        solver = solver_cls()
        dataset_reader = dataset_reader_cls(os.path.join(dataset_path, dataset_name))

        return solver, dataset_reader
