import math
import torch as ch

def print_message_pvals(p):
    if p < 0.05:
        print(f'Pval: {p:.3f}. Reject null (the 2 distributions are not the same).')
    else:
        print(f'Pval: {p:.3f}. 2 distributions are not distinguishable.')

def rgb_to_gray(imgs):
    """
    Convert images from RGB to grayscale (each pixel is represented by its luminance)
    Formula used in 1D-tests paper: 0.299*R + 0.587*G + 0.114*B
    https://arxiv.org/abs/1708.02688
    Arguments
    ---------
    imgs : np.array
        A batch of RGB images

    Returns
    -------
    np.array
        A batch of grayscale images
    """
    return 0.299*imgs[:,0:1,:,:] + 0.587*imgs[:,1:2,:,:] + 0.114*imgs[:,2:3,:,:]

def skew(x): 
    x = x.view(x.size(0),-1)
    N = x.size(1)
    mean = x.mean(1).unsqueeze(1)
    x_ = (x - mean)
    m3 = (x_**3).mean(1)
    m2 = (x_**2).mean(1)
    
    coef = math.sqrt(N*(N-1))/(N-2)
    return coef*m3/(m2**1.5)

def kurtosis(x):
    N = x.size(1)
    mu = x.mean(1).unsqueeze(1)
    x_ = (x-mu)
    m4 = (x_**4).mean(1)
    m2 = (x_**2).mean(1)
    coef = (N-1)/((N-2)*(N-3))
    return coef*((N+1)*m4/(m2**2) - 3*(N-1))

def ttest_ind(x1,x2): 
    n1,n2 = x1.size(0),x2.size(0)
    mu1, mu2 = x1.mean(), x2.mean()
    std1, std2 = x1.std()/math.sqrt(n1), x2.std()/math.sqrt(n2)
    return (mu1 - mu2)/ch.sqrt(std1**2 + std2**2)

