import torch.nn as nn
import torch
from modules import OnlineNeuron
from resnet import conv3x3, ECAAttention

cfg = {
    'VGG11': [
        [64],
        [128, 'M'],
        [256, 256, 'M'],
        [512, 512, 'M'],
        [512, 512, 'M']
    ],
    'VGG13': [
        [64],
        [64, 'M', 128, 128, 'M'],
        [256, 256, 'M'],
        [512, 512, 512, 'M'],
        [512, 'MM']
    ],
    'VGG16': [
        [64],
        [64, 'M', 128, 128, 'M'],
        [256, 256, 256, 'M'],
        [512, 512, 512, 'M'],
        [512, 512, 512, 'M']
    ],
    'VGG19': [
        [64],
        [64, 'M', 128, 128, 'M'],
        [256, 256, 256, 256, 'M'],
        [512, 512, 512, 512, 'M'],
        [512, 512, 512, 512, 'M']
    ]
}


class VGG(nn.Module):
    def __init__(self, T, vgg_name, num_classes, use_dvs, use_eca, use_mem_bn):
        super(VGG, self).__init__()
        self.init_channels = 3 if use_dvs is False else 2
        self.T = T
        self.use_eca = use_eca
        self.use_mem_bn = use_mem_bn
        self.layer1 = self._make_layers(cfg[vgg_name][0], True if use_dvs is False else False)
        self.layer2 = self._make_layers(cfg[vgg_name][1])
        self.layer3 = self._make_layers(cfg[vgg_name][2])
        self.layer4 = self._make_layers(cfg[vgg_name][3])
        self.layer5 = self._make_layers(cfg[vgg_name][4])

        if use_dvs is True:
            self.classifier = nn.Linear(512*3*3, num_classes)   
        else:
            self.classifier = nn.Linear(512, num_classes)     

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.constant_(m.weight, val=1)
                nn.init.zeros_(m.bias)
            elif isinstance(m, nn.Linear):
                nn.init.zeros_(m.bias)

    def _make_layers(self, cfg, first_conv=False):
        layers = []
        for x in cfg:
            if x == 'M':
                #layers.append(nn.MaxPool2d(kernel_size=2, stride=2))
                layers.append(nn.AvgPool2d(kernel_size=2, stride=2))
            elif x == 'MM':
                layers.append(nn.AdaptiveAvgPool2d((1, 1)))
            else:
                if first_conv is True:
                    layers.append(nn.Conv2d(self.init_channels, x, kernel_size=3, padding=1, bias=False))
                else:
                    layers.append(conv3x3(self.init_channels, x))
                layers.append(nn.BatchNorm2d(x))
                if self.use_eca > 0:
                    layers.append(ECAAttention())
                layers.append(OnlineNeuron(self.T, self.use_mem_bn, x))
                self.init_channels = x
        return nn.Sequential(*layers)

    def forward(self, x):       
        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)
        x = self.layer5(x)
        x = torch.flatten(x, 1)
        return self.classifier(x)


def vgg11(T, num_classes, use_dvs, use_eca, use_mem_bn):
    return VGG(T, 'VGG11', num_classes, use_dvs, use_eca, use_mem_bn)

def vgg13(T, num_classes, use_dvs, use_eca, use_mem_bn):
    return VGG(T, 'VGG13', num_classes, use_dvs, use_eca, use_mem_bn)

def vgg16(T, num_classes, use_dvs, use_eca, use_mem_bn):
    return VGG(T, 'VGG16', num_classes, use_dvs, use_eca, use_mem_bn)

def vgg19(T, num_classes, use_dvs, use_eca, use_mem_bn):
    return VGG(T, 'VGG19', num_classes, use_dvs, use_eca, use_mem_bn)