DEFAULT_IMPROVEMENT_GUIDANCE = """It can be a simple bug fix, the introduction of a new algorithm, or any degree of change from minor to major.
Use your experience from previous attempts to develop the best approach possible."""


def get_improvement_guidance_with_weights(prompt_enforce=""):
    """Returns various improvement guidance and their weights.

    Args:
        prompt_enforce: A prompt that should be enforced in the response.

    Returns:
        The list of tuples, where each tuple contains a improvement guidance and its weight.
    """

    improvement_guidance_with_weights = [
        (
            # Improvement guidance for general cases
            f"It can be a simple bug fix, the introduction of a new algorithm, or any degree of change from minor to major. "
            f"You don't need to implement the solution yet. Instead, please think deeply and broadly about the possible improvements, and explain your thoughts. "
            f"{prompt_enforce}",
            0.2,
        ),
        (
            # Improvement guidance for complexity analysis
            f"Based on the code and feedback: What are the key algorithms and data structures used? "
            f"What are its computational complexity bottlenecks (consider feedback like Time Limit Exceeded)? "
            f"How might we improve the time or space complexity (consider feedback like Memory Limit Exceeded)? "
            f"Consider both small optimizations and completely different approaches. "
            f"You don't need to implement the solution yet. Instead, please think deeply and broadly about the possible improvements, and explain your thoughts. "
            f"{prompt_enforce}",
            0.2,
        ),
        (
            # Improvement guidance for simulated annealing
            f"If this solution uses simulated annealing, analyze the problem, solution properties, and feedback to suggest better state representation. "
            f"Consider how the current state encoding might be limiting the search space or convergence speed, especially in light of the feedback. "
            f"Think about alternative state encodings that could lead to better local optima or faster convergence. "
            f"You don't need to implement the solution yet. Instead, please think deeply and broadly about the possible improvements, and explain your thoughts. "
            f"{prompt_enforce}",
            0.2,
        ),
        (
            # Improvement guidance for simulated annealing neighborhood design
            f"If this solution uses simulated annealing, analyze the problem, solution properties, and feedback to suggest better neighborhood design. "
            f"Consider how the current neighborhood structure might be limiting the search space exploration or convergence speed, based on the feedback. "
            f"Think about alternative neighborhood structures that could lead to better local optima or faster convergence. "
            f"Specifically, consider: "
            f"1. How to balance between small and large moves in the search space "
            f"2. How to ensure the neighborhood structure allows reaching any valid solution "
            f"3. How to design moves that maintain solution feasibility while exploring new regions "
            f"You don't need to implement the solution yet. Instead, please think deeply and broadly about the possible improvements, and explain your thoughts. "
            f"{prompt_enforce}",
            0.2,
        ),
        (
            # Improvement guidance for beam search
            f"Consider implementing or improving a beam search approach. "
            f"Think about the beam width and evaluation function that could lead to better solutions. "
            f"Consider how to effectively balance between diversity and quality in your beam. "
            f"You don't need to implement the solution yet. Instead, please think deeply and broadly about the possible improvements, and explain your thoughts. "
            f"{prompt_enforce}",
            0.2,
        ),
    ]

    return improvement_guidance_with_weights


def get_message_templates_with_weights(code_block_format, parent_code, parent_feedback_str, prompt_enforce=""):
    """Returns various prompt templates and their weights.

    Args:
        code_block_format: The format for the code block.
        parent_code: The code from the parent node.
        parent_feedback_str: The feedback string from the parent node.
        prompt_enforce: A prompt that should be enforced in the response.

    Returns:
        The list of tuples, where each tuple contains a prompt template and its weight.
    """
    # Get the improvement guidance with weights
    improvement_guidance_list = get_improvement_guidance_with_weights(prompt_enforce)

    message_templates_with_weights = []

    # Common prefix for the prompt
    common_prefix = [
        (
            # Original
            f"The following is a previous implementation of the solution:\n"
            f"{code_block_format.format(parent_code)}\n\n"
            f"Here is the feedback for this implementation:\n{parent_feedback_str}\n\n"
            f"Please analyze the code and the feedback, and list what insights can be gained. "
            f"Then, based on the insights, please consider the possible improvements to the code. ",
            0.5,
        ),
        (
            # Another introduction pattern
            f"Examine this existing solution to the problem:\n"
            f"{code_block_format.format(parent_code)}\n\n"
            f"Here is the feedback for this implementation:\n{parent_feedback_str}\n\n"
            f"Based on the code and feedback: ",
            0.5,
        ),
    ]

    # Create a list of tuples (template combined with prefix, weight)
    for prefix, prefix_weight in common_prefix:
        for guidance, guidance_weight in improvement_guidance_list:
            # Recalculate weight (multiply both weights)
            combined_weight = prefix_weight * guidance_weight
            # Create the full template
            full_template = prefix + guidance
            # Add to the list
            message_templates_with_weights.append((full_template, combined_weight))

    # Normalize weights to sum to 1
    total_weight = sum(weight for _, weight in message_templates_with_weights)
    message_templates_with_weights = [
        (template, weight / total_weight) for template, weight in message_templates_with_weights
    ]

    return message_templates_with_weights


# Template for feedback with summary
FEEDBACK_WITH_SUMMARY_TEMPLATE = """
[Summary of your previous attempts]
{action_summary}

[Your best submission]
### Code
```{lang}
{best_code}
```

### Feedback
{best_feedback}

[Your latest submission]
### Code
```{lang}
{latest_code}
```

### Feedback
{latest_feedback}

Based on the above feedback, please consider ways to improve your solution.
First, analyze this feedback and list what insights can be gained from it.
Also, please create a new summary including the content of your previous attempts in Markdown format in the ```md
<!-- Your summary here -->
``` code block.
If this code block format is not found, the summary of your previous attempts will not be included in the next turn.

Then, based on these insights, please refine your code to achieve better performance.
{improvement_guidance}
"""
