import argparse
import datetime
import json
from pathlib import Path
import shutil

from ale_bench.result import CaseResult, JudgeResult, Result


def parse_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser(description="Aggregate results from multiple runs.")
    parser.add_argument(
        "--root_dir",
        type=Path,
        default=Path(__file__).resolve().parent / "results",
        help="Root directory containing the results.",
    )
    parser.add_argument(
        "--target_dir",
        type=Path,
        default=Path(__file__).resolve().parent / "results_new",
        help="Root directory to save the results.",
    )
    return parser.parse_args()


PROBLEM_IDS = open(Path(__file__).resolve().parents[1] / "problem_ids.txt").read().splitlines()


def replace_details_in_case_result(result: Result) -> Result:
    processed_case_results = []
    need_details = False
    if result.overall_judge_result != JudgeResult.ACCEPTED:
        need_details = True
    for case_result in result.case_results:
        if need_details and case_result.judge_result == result.overall_judge_result:
            processed_case_results.append(CaseResult(
                input_str=case_result.input_str,
                output_str=case_result.output_str,
                error_str=case_result.error_str,
                judge_result=case_result.judge_result,
                message=case_result.message,
                absolute_score=case_result.absolute_score,
                relative_score=case_result.relative_score,
                local_visualization=case_result.local_visualization,
                execution_time=case_result.execution_time,
                memory_usage=case_result.memory_usage,
            ))
            need_details = False
        else:
            processed_case_results.append(CaseResult(
                input_str=None,
                output_str=None,
                error_str=None,
                judge_result=case_result.judge_result,
                message=case_result.message,
                absolute_score=case_result.absolute_score,
                relative_score=case_result.relative_score,
                local_visualization=None,
                execution_time=case_result.execution_time,
                memory_usage=case_result.memory_usage,
            ))
    return Result(
        allow_score_non_ac=result.allow_score_non_ac,
        resource_usage=result.resource_usage,
        case_results=processed_case_results,
    )


def main(root_dir: Path, target_dir: Path) -> None:
    for subdir in root_dir.iterdir():
        if subdir.name.startswith("llm_log"):
            continue  # Skip LLM log directories
        if subdir.is_dir():
            main(subdir, target_dir / subdir.name)
    private_results = list(root_dir.glob("private_result_*.json"))
    if len(private_results) == 0:
        return
    for file in private_results:
        if not file.is_file():
            continue
        _, problem_id = file.stem.rsplit("_", 1)
        assert (root_dir / f"llm_log_{problem_id}").is_dir(), f"Missing `llm_log` directory for {problem_id} ({target_dir.name})"
        assert (root_dir / f"codes_history_{problem_id}.json").is_file(), f"Missing `codes_history` file for {problem_id} ({target_dir.name})"
        assert (root_dir / f"llm_history_{problem_id}.json").is_file(), f"Missing `llm_history` file for {problem_id} ({target_dir.name})"
        assert (root_dir / f"log_{problem_id}.txt").is_file(), f"Missing `log` file for {problem_id} ({target_dir.name})"
        assert (root_dir / f"session_{problem_id}.json").is_file(), f"Missing `session` file for {problem_id} ({target_dir.name})"
        shutil.copytree(root_dir / f"llm_log_{problem_id}", target_dir / f"llm_log_{problem_id}")
        codes_history = json.load((root_dir / f"codes_history_{problem_id}.json").open("r"))
        codes_history = [[
            replace_details_in_case_result(Result.model_validate(r)).model_dump(), l, c
        ] for r, l, c in codes_history]
        json.dump(codes_history,(target_dir / f"codes_history_{problem_id}.json").open("w"))
        shutil.copyfile(root_dir / f"llm_history_{problem_id}.json", target_dir / f"llm_history_{problem_id}.json")
        shutil.copyfile(root_dir / f"log_{problem_id}.txt", target_dir / f"log_{problem_id}.txt")
        shutil.copyfile(file, target_dir / file.name)
        shutil.copyfile(root_dir / f"session_{problem_id}.json", target_dir / f"session_{problem_id}.json")

if __name__ == "__main__":
    args = parse_args()
    main(args.root_dir, args.target_dir)
