import argparse
import collections
import json
import os
import random
import sys
import time
import copy
import uuid
import numpy as np
import PIL
import torch
import torchvision
import torch.utils.data
import yaml
import datasets
import hparams_registry
import algorithms
import numpy.random as random
from lib import misc
from lib.fast_data_loader import InfiniteDataLoader, FastDataLoader
def get_config(config):
    with open(config, 'r') as stream:
        return yaml.load(stream)

def set_gpu(x):
    os.environ['CUDA_VISIBLE_DEVICES'] = x
    print('using gpu:', x)

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Domain generalization')
    parser.add_argument('--data_dir', type=str, default='./data')
    parser.add_argument('--dataset', type=str, default="AFAD")
    parser.add_argument('--algorithm', type=str, default="ERM")
    parser.add_argument('--gen_dir', type=str, default="", help="if not empty, the generator of DEDF will be loaded")
    parser.add_argument('--stage', type=int, default=1,
        help='hyperparameter for DDG, 0:train the gan, 1: train the model')
    parser.add_argument('--gpu_id', 
                      help='gpu device number', 
                      type=str, default='1')
    parser.add_argument('--task', type=str, default="domain_generalization",
        help='domain_generalization | domain_adaptation')
    parser.add_argument('--hparams', type=str,
        help='JSON-serialized hparams dict')
    parser.add_argument('--hparams_seed', type=int, default=0,
        help='Seed for random hparams (0 means "default hparams")')
    parser.add_argument('--image_display_iter', type=int, default=500,
        help='Epochs interval for showing the generated images')
    parser.add_argument('--trial_seed', type=int, default=0,
        help='Trial number (used for seeding split_dataset and '
        'random_hparams).')
    parser.add_argument('--seed', type=int, default=7,
        help='Seed for everything else')
    parser.add_argument('--steps', type=int, default=None,
        help='Number of steps. Default is dataset-dependent.')
    parser.add_argument('--checkpoint_freq', type=int, default=None,
        help='Checkpoint every N steps. Default is dataset-dependent.')
    parser.add_argument('--test_envs', type=int, nargs='+', default=[4])
    parser.add_argument('--output_dir', type=str, default="log")
    parser.add_argument('--holdout_fraction', type=float, default=0.1)
    parser.add_argument('--uda_holdout_fraction', type=float, default=0)
    parser.add_argument('--skip_model_save', action='store_true')
    parser.add_argument('--save_model_every_checkpoint', action='store_true')
    args = parser.parse_args()


    set_gpu(args.gpu_id)
    

    # If we ever want to implement checkpointing, just persist these values
    # every once in a while, and then load them from disk here.
    start_step = 0
    algorithm_dict = None

    args.output_dir=args.output_dir+"/_ds_"+args.dataset+"_alg_"+args.algorithm+"_seed_"+ str(args.trial_seed)+"_test_"+ str(args.test_envs)

    jsfile=os.path.join(args.output_dir,"results.jsonl")
    

    os.makedirs(args.output_dir, exist_ok=True)
    sys.stdout = misc.Tee(os.path.join(args.output_dir, 'out.txt'))
    sys.stderr = misc.Tee(os.path.join(args.output_dir, 'err.txt'))

    print("Environment:")
    print("\tPython: {}".format(sys.version.split(" ")[0]))
    print("\tPyTorch: {}".format(torch.__version__))
    print("\tTorchvision: {}".format(torchvision.__version__))
    print("\tCUDA: {}".format(torch.version.cuda))
    print("\tCUDNN: {}".format(torch.backends.cudnn.version()))
    print("\tNumPy: {}".format(np.__version__))
    print("\tPIL: {}".format(PIL.__version__))

    print('Args:')
    for k, v in sorted(vars(args).items()):
        print('\t{}: {}'.format(k, v))

    if args.hparams_seed == 0:
        hparams = hparams_registry.default_hparams(args.algorithm, args.dataset, stage=args.stage)
    else:
        hparams = hparams_registry.random_hparams(args.algorithm, args.dataset,
            misc.seed_hash(args.hparams_seed, args.trial_seed), stage=args.stage)
    if args.hparams:
        hparams.update(json.loads(args.hparams))


    for k, v in sorted(hparams.items()):
        print('\t{}: {}'.format(k, v))

    random.seed(args.trial_seed)
    np.random.seed(args.trial_seed)
    torch.manual_seed(args.trial_seed)
    torch.cuda.manual_seed(args.trial_seed)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False

    if torch.cuda.is_available():
        device = "cuda"
    else:
        device = "cpu"

    if args.dataset in vars(datasets):
        dataset = vars(datasets)[args.dataset](args.data_dir,
            args.test_envs, hparams)
    else:
        raise NotImplementedError

    # Split each env into an 'in-split' and an 'out-split'. We'll train on
    # each in-split except the test envs, and evaluate on all splits.
    
    # To allow unsupervised domain adaptation experiments, we split each test
    # env into 'in-split', 'uda-split' and 'out-split'. The 'in-split' is used
    # by collect_results.py to compute classification accuracies.  The
    # 'out-split' is used by the Oracle model selectino method. The unlabeled
    # samples in 'uda-split' are passed to the algorithm at training time if
    # args.task == "domain_adaptation". If we are interested in comparing
    # domain generalization and domain adaptation results, then domain
    # generalization algorithms should create the same 'uda-splits', which will
    # be discared at training.
    in_splits = []
    out_splits = []
    uda_splits = []
    for env_i, env in enumerate(dataset):
        uda = []
        out, in_ = misc.split_dataset(env,
            int(len(env)*args.holdout_fraction),
            misc.seed_hash(args.trial_seed, env_i))

        if env_i in args.test_envs:
            uda, in_ = misc.split_dataset(in_,
                int(len(in_)*args.uda_holdout_fraction),
                misc.seed_hash(args.trial_seed, env_i))

        if hparams['class_balanced']:
            in_weights = misc.make_weights_for_balanced_classes(in_)
            out_weights = misc.make_weights_for_balanced_classes(out)
            if uda is not None:
                uda_weights = misc.make_weights_for_balanced_classes(uda)
        else:
            in_weights, out_weights, uda_weights = None, None, None


        in_splits.append((in_, in_weights))
        out_splits.append((out, out_weights))
        if len(uda):
            uda_splits.append((uda, uda_weights))

    train_loaders = [InfiniteDataLoader(
        dataset=env,
        weights=env_weights,
        batch_size=hparams['batch_size'],
        num_workers=dataset.N_WORKERS)
        for i, (env, env_weights) in enumerate(in_splits)
        if i not in args.test_envs]
    
    uda_loaders = [InfiniteDataLoader(
        dataset=env,
        weights=env_weights,
        batch_size=hparams['batch_size'],
        num_workers=dataset.N_WORKERS)
        for i, (env, env_weights) in enumerate(uda_splits)
        if i in args.test_envs]

    eval_loaders = [FastDataLoader(
        dataset=env,
        batch_size=64,
        num_workers=dataset.N_WORKERS)
        for env, _ in (in_splits + out_splits + uda_splits)]
        
    eval_weights = [None for _, weights in (in_splits + out_splits + uda_splits)]
    eval_loader_names = ['env{}_in'.format(i)
        for i in range(len(in_splits))]
    eval_loader_names += ['env{}_out'.format(i)
        for i in range(len(out_splits))]
    eval_loader_names += ['env{}_uda'.format(i)
        for i in range(len(uda_splits))]

    algorithm_class = algorithms.get_algorithm_class(args.algorithm)
    algorithm = algorithm_class(dataset.input_shape, dataset.num_classes,
        len(dataset) - len(args.test_envs), hparams)

    if algorithm_dict is not None:
        algorithm.load_state_dict(algorithm_dict)

    algorithm.to(device)
    if args.algorithm == 'DDG' and args.gen_dir and hparams['stage'] == 1:
        pretext_model = torch.load(args.gen_dir)['model_dict']
        alg_dict = algorithm.state_dict()
        ignored_keys = []
        state_dict = {k: v for k, v in pretext_model.items() if k in alg_dict.keys() and ('id_featurizer' in k or 'gen' in k)}
        alg_dict.update(state_dict)
        algorithm.load_state_dict(alg_dict)
        algorithm_copy = copy.deepcopy(algorithm)
        algorithm_copy.eval()
    else:
        algorithm_copy = None

    train_minibatches_iterator = zip(*train_loaders)
    uda_minibatches_iterator = zip(*uda_loaders)
    checkpoint_vals = collections.defaultdict(lambda: [])

    steps_per_epoch = min([len(env)/hparams['batch_size'] for env,_ in in_splits]) if args.algorithm != 'DDG' else min([len(env)/hparams['batch_size']/2 for env,_ in in_splits])
    print("steps per epoch", steps_per_epoch)

    n_steps = args.steps or dataset.N_STEPS
    if 'DDG' in args.algorithm:
        n_steps = hparams['steps']
    checkpoint_freq = args.checkpoint_freq or dataset.CHECKPOINT_FREQ

    def save_checkpoint(filename):
        save_dict = {
            "args": vars(args),
            "model_input_shape": dataset.input_shape,
            "model_num_classes": dataset.num_classes,
            "model_num_domains": len(dataset) - len(args.test_envs),
            "model_hparams": hparams,
            "model_dict": algorithm.state_dict()
        }
        torch.save(save_dict, os.path.join(args.output_dir, filename))


    last_results_keys = None
    print("n_steps", n_steps)
    best_val_mse=100000000
    cnt=1

    for step in range(start_step, n_steps):
        step_start_time = time.time()
        if args.task == "domain_adaptation":
            uda_device = [x.to(device)
                for x,_ in next(uda_minibatches_iterator)]
        else:
            uda_device = None

        if 'DDG' in args.algorithm:
            minibatches_device =  [(x.to(device).float(), y.to(device).float(), pos.to(device)) for x,y,pos in next(train_minibatches_iterator)]
            minibatches_device_neg = [(x.to(device).float(), y.to(device).float(), pos.to(device)) for x,y,pos in next(train_minibatches_iterator)]
            step_vals =  algorithm.update(minibatches_device, minibatches_device_neg,  pretrain_model=algorithm_copy)
        else:
            minibatches_device = [(x.to(device).float(), y.to(device).float()) for x,y in next(train_minibatches_iterator)]  
            step_vals =  algorithm.update(minibatches_device,  uda_device)
        checkpoint_vals['step_time'].append(time.time() - step_start_time)

        for key, val in step_vals.items():
            checkpoint_vals[key].append(val)
        if not os.path.exists(os.path.join(args.output_dir,'images')):
            os.makedirs(os.path.join(args.output_dir,'images'))
        if (step % checkpoint_freq == 0) or (step == n_steps - 1):
            results = {
                'step': step,
                'epoch': step / steps_per_epoch,
            }

            for key, val in checkpoint_vals.items():
                results[key] = np.mean(val)

            evals = zip(eval_loader_names, eval_loaders, eval_weights)
            for name, loader, weights in evals:
                mse_loss = misc.accuracy(algorithm, loader, weights, device, args=args, step=step, is_ddg=hparams['is_ddg'])
                results[name+'_mse'] = mse_loss

            IID_mse=0
            # min_mse=10111111

            for key in results.keys():
                if "out_mse" in key and (not key == "env{}_out_mse".format(args.test_envs[0])):
                    IID_mse+=results[key]
            
            if IID_mse < best_val_mse:
                    best_val_mse = IID_mse
                    results["***env{}_in_mse_IID_BEST***".format(args.test_envs[0])] = results["env{}_in_mse".format(args.test_envs[0])]
                    cnt=0
            results_keys = sorted(results.keys())
            if results_keys != last_results_keys:
                misc.print_row(results_keys, colwidth=12)
                last_results_keys = results_keys
            misc.print_row([results[key] for key in results_keys],
                colwidth=12)

            results.update({
                'hparams': hparams,
                'args': vars(args)    
            })

            epochs_path = os.path.join(args.output_dir, 'results.jsonl')
            with open(epochs_path, 'a') as f:
                f.write(json.dumps(results, sort_keys=True) + "\n")

            start_step = step + 1
            checkpoint_vals = collections.defaultdict(lambda: [])



