try:
    # until python 3.10
    from collections import Mapping
except:
    # from python 3.10
    from collections.abc import Mapping
from copy import deepcopy
import os
from os.path import dirname, abspath
import sys
import yaml
import ast

import numpy as np
from sacred import Experiment, SETTINGS
from sacred.observers import FileStorageObserver
from sacred.utils import apply_backspaces_and_linefeeds
import torch as th

from utils.logging import get_logger
from run_llm_final import run
from absl import logging

SETTINGS["CAPTURE_MODE"] = (
    "fd"  # set to "no" if you want to see stdout/stderr in console
)
logger = get_logger()
logging.set_verbosity(logging.WARN)

ex = Experiment("pymarl")
ex.logger = logger
ex.captured_out_filter = apply_backspaces_and_linefeeds

results_path = os.path.join(dirname(dirname(abspath(__file__))), "results")
# results_path = "/home/ubuntu/data"

@ex.main
def my_main(_run, _config, _log):
    # Setting the random seed throughout the modules
    config = config_copy(_config)
    np.random.seed(config["seed"])
    th.manual_seed(config["seed"])
    config["env_args"]["seed"] = config["seed"]

    # run the framework
    run(_run, config, _log)


def _get_config(params, arg_name, subfolder):
    config_name = None
    for _i, _v in enumerate(params):
        if _v.split("=")[0] == arg_name:
            config_name = _v.split("=")[1]
            del params[_i]
            break

    if config_name is not None:
        with open(
            os.path.join(
                os.path.dirname(__file__),
                "config",
                subfolder,
                "{}.yaml".format(config_name),
            ),
            "r",
        ) as f:
            try:
                config_dict = yaml.load(f, Loader=yaml.FullLoader)
            except yaml.YAMLError as exc:
                assert False, "{}.yaml error: {}".format(config_name, exc)
        return config_dict


def recursive_dict_update(d, u):
    for k, v in u.items():
        if isinstance(v, Mapping):
            d[k] = recursive_dict_update(d.get(k, {}), v)
        else:
            d[k] = v
    return d


def config_copy(config):
    if isinstance(config, dict):
        return {k: config_copy(v) for k, v in config.items()}
    elif isinstance(config, list):
        return [config_copy(v) for v in config]
    else:
        return deepcopy(config)

def main(params, cuda_visible_device):
    import torch
    num_devices = torch.cuda.device_count()
    torch.cuda.set_device(cuda_visible_device % num_devices)

    params = deepcopy(params)
    th.set_num_threads(1)

    # Get the defaults from default.yaml
    with open(
        os.path.join(os.path.dirname(__file__), "config", "default.yaml"), "r"
    ) as f:
        try:
            config_dict = yaml.load(f, Loader=yaml.FullLoader)
        except yaml.YAMLError as exc:
            assert False, "default.yaml error: {}".format(exc)

    # Load algorithm and env base configs
    env_config = _get_config(params, "--env-config", "envs")
    alg_config = _get_config(params, "--config", "algs")
    config_dict = recursive_dict_update(config_dict, env_config)
    config_dict = recursive_dict_update(config_dict, alg_config)

    for param in params:
        if param.startswith("comm_code_path"):
            comm_code_path = param.split("=")[1]
            config_dict["comm_code_path"] = comm_code_path
        if param.startswith("comm_code_path_add"):
            comm_code_path_add = param.split("=")[1]
            config_dict["comm_code_path_add"] = comm_code_path_add
        if param.startswith("comm_code_path_updated_timewise"):
                    comm_code_path_updated_timewise = param.split("=")[1]
                    config_dict["comm_code_path_updated_timewise"] = comm_code_path_updated_timewise
        if param.startswith("important_state"):
            important_state = param.split("=")[1]
            try:
                config_dict["important_state"] = ast.literal_eval(important_state)
            except Exception:
                config_dict["important_state"] = important_state
        if param.startswith("phase"):
            config_dict["phase"] = param.split("=")[1]
        if param.startswith("seed"):
            config_dict["seed"] = int(param.split("=")[1])
        if param.startswith("env_args.map_name"):
            map_name = param.split("=")[1]
        elif param.startswith("env_args.key"):
            map_name = param.split("=")[1]

    # now add all the config to sacred
    ex.add_config(config_dict)

    # Save to disk by default for sacred
    logger.info("Saving to FileStorageObserver in results/sacred.")
    try:
        result_name = config_dict['name']
    except:
        result_name = "unnamed"
    try:
        file_obs_path = os.path.join(
            results_path, f"sacred/{result_name}/{map_name}"
        )
    except:
        file_obs_path = os.path.join(results_path, f"sacred/{result_name}")

    ex.observers.append(FileStorageObserver.create(file_obs_path))

    ex.run_commandline(params)
