import numpy as np

np.random.seed(0)

def draw_unit_cube_numpy(K, d=10):
    return np.random.rand(K, d)

def compute_weights_looped(Y, g_opt, K, loops=10):
    d = Y.shape[1]
    J = Y.shape[0]
    weights_total = np.zeros(J)

    for _ in range(loops):
        X = draw_unit_cube_numpy(K, d)  # shape (K, d)

        # Compute cost(x, y_j) = 0.5 * ||x - y_j||^2 - g_j
        X_sq = np.sum(X**2, axis=1, keepdims=True)       # (K, 1)
        Y_sq = np.sum(Y**2, axis=1)                       # (J,)
        XY = X @ Y.T                                      # (K, J)
        distances = 0.5 * (X_sq - 2 * XY + Y_sq) - g_opt  # (K, J)

        indices = np.argmin(distances, axis=1)            # (K,)
        weights = np.bincount(indices, minlength=J)       # (J,)
        weights_total += weights

    return weights_total / (K * loops)

# Parameters
J = 100
d = 10
K = 10**6
loops = 10

# Step 1: Initial draw
Y = draw_unit_cube_numpy(J, d)
g_opt = np.random.rand(J)

# Step 2: First weights
weights = compute_weights_looped(Y, g_opt, K, loops)

# Step 3: Drop small weights
mask = weights > 0.1 / J
Y = Y[mask]
g_opt = g_opt[mask]

print("J now:" + str(Y.shape[0]))
# Step 4: Recompute weights
weights = compute_weights_looped(Y, g_opt, K*10, loops)

# Output
print("Final J:", Y.shape[0])
print("Weights:", weights)
print("g_opt:", g_opt)

# Save to files
np.save('nu_weights_ex2.npy', weights)
np.save('nu_points_ex2.npy', Y)
np.save('g_opt_ex2.npy', g_opt)

