import functools
import os
import site
import sys
import sysconfig
import typing

from pip._internal.exceptions import InstallationError
from pip._internal.utils import appdirs
from pip._internal.utils.virtualenv import running_under_virtualenv


USER_CACHE_DIR = appdirs.user_cache_dir("pip")


site_packages: typing.Optional[str] = sysconfig.get_path("purelib")


def get_major_minor_version() -> str:
    
    return "{}.{}".format(*sys.version_info)


def change_root(new_root: str, pathname: str) -> str:
    
    if os.name == "posix":
        if not os.path.isabs(pathname):
            return os.path.join(new_root, pathname)
        else:
            return os.path.join(new_root, pathname[1:])

    elif os.name == "nt":
        (drive, path) = os.path.splitdrive(pathname)
        if path[0] == "\\":
            path = path[1:]
        return os.path.join(new_root, path)

    else:
        raise InstallationError(
            f"Unknown platform: {os.name}\n"
            "Can not change root path prefix on unknown platform."
        )


def get_src_prefix() -> str:
    if running_under_virtualenv():
        src_prefix = os.path.join(sys.prefix, "src")
    else:
        
        try:
            src_prefix = os.path.join(os.getcwd(), "src")
        except OSError:
            
            sys.exit("The folder you are executing pip from can no longer be found.")

    
    
    return os.path.abspath(src_prefix)


try:
    
    
    user_site: typing.Optional[str] = site.getusersitepackages()
except AttributeError:
    user_site = site.USER_SITE


@functools.lru_cache(maxsize=None)
def is_osx_framework() -> bool:
    return bool(sysconfig.get_config_var("PYTHONFRAMEWORK"))
