



import hashlib
import os
from textwrap import dedent

from ..cache import BaseCache, SeparateBodyBaseCache
from ..controller import CacheController

try:
    FileNotFoundError
except NameError:
    
    FileNotFoundError = (IOError, OSError)


def _secure_open_write(filename, fmode):
    
    flags = os.O_WRONLY

    
    
    
    
    flags |= os.O_CREAT | os.O_EXCL

    
    
    if hasattr(os, "O_NOFOLLOW"):
        flags |= os.O_NOFOLLOW

    
    if hasattr(os, "O_BINARY"):
        flags |= os.O_BINARY

    
    
    try:
        os.remove(filename)
    except (IOError, OSError):
        
        pass

    
    
    
    
    fd = os.open(filename, flags, fmode)
    try:
        return os.fdopen(fd, "wb")

    except:
        
        os.close(fd)
        raise


class _FileCacheMixin:
    

    def __init__(
        self,
        directory,
        forever=False,
        filemode=0o0600,
        dirmode=0o0700,
        use_dir_lock=None,
        lock_class=None,
    ):

        if use_dir_lock is not None and lock_class is not None:
            raise ValueError("Cannot use use_dir_lock and lock_class together")

        try:
            from lockfile import LockFile
            from lockfile.mkdirlockfile import MkdirLockFile
        except ImportError:
            notice = dedent(
                
            )
            raise ImportError(notice)

        else:
            if use_dir_lock:
                lock_class = MkdirLockFile

            elif lock_class is None:
                lock_class = LockFile

        self.directory = directory
        self.forever = forever
        self.filemode = filemode
        self.dirmode = dirmode
        self.lock_class = lock_class

    @staticmethod
    def encode(x):
        return hashlib.sha224(x.encode()).hexdigest()

    def _fn(self, name):
        
        
        hashed = self.encode(name)
        parts = list(hashed[:5]) + [hashed]
        return os.path.join(self.directory, *parts)

    def get(self, key):
        name = self._fn(key)
        try:
            with open(name, "rb") as fh:
                return fh.read()

        except FileNotFoundError:
            return None

    def set(self, key, value, expires=None):
        name = self._fn(key)
        self._write(name, value)

    def _write(self, path, data: bytes):
        
        
        try:
            os.makedirs(os.path.dirname(path), self.dirmode)
        except (IOError, OSError):
            pass

        with self.lock_class(path) as lock:
            
            with _secure_open_write(lock.path, self.filemode) as fh:
                fh.write(data)

    def _delete(self, key, suffix):
        name = self._fn(key) + suffix
        if not self.forever:
            try:
                os.remove(name)
            except FileNotFoundError:
                pass


class FileCache(_FileCacheMixin, BaseCache):
    

    def delete(self, key):
        self._delete(key, "")


class SeparateBodyFileCache(_FileCacheMixin, SeparateBodyBaseCache):
    

    def get_body(self, key):
        name = self._fn(key) + ".body"
        try:
            return open(name, "rb")
        except FileNotFoundError:
            return None

    def set_body(self, key, body):
        name = self._fn(key) + ".body"
        self._write(name, body)

    def delete(self, key):
        self._delete(key, "")
        self._delete(key, ".body")


def url_to_file_path(url, filecache):
    
    key = CacheController.cache_url(url)
    return filecache._fn(key)
