from __future__ import annotations

import os
import re
import sys
from functools import lru_cache
from typing import cast

from .api import PlatformDirsABC


class Android(PlatformDirsABC):
    

    @property
    def user_data_dir(self) -> str:
        
        return self._append_app_name_and_version(cast(str, _android_folder()), "files")

    @property
    def site_data_dir(self) -> str:
        
        return self.user_data_dir

    @property
    def user_config_dir(self) -> str:
        
        return self._append_app_name_and_version(cast(str, _android_folder()), "shared_prefs")

    @property
    def site_config_dir(self) -> str:
        
        return self.user_config_dir

    @property
    def user_cache_dir(self) -> str:
        
        return self._append_app_name_and_version(cast(str, _android_folder()), "cache")

    @property
    def user_state_dir(self) -> str:
        
        return self.user_data_dir

    @property
    def user_log_dir(self) -> str:
        
        path = self.user_cache_dir
        if self.opinion:
            path = os.path.join(path, "log")
        return path

    @property
    def user_documents_dir(self) -> str:
        
        return _android_documents_folder()

    @property
    def user_runtime_dir(self) -> str:
        
        path = self.user_cache_dir
        if self.opinion:
            path = os.path.join(path, "tmp")
        return path


@lru_cache(maxsize=1)
def _android_folder() -> str | None:
    
    try:
        
        from jnius import autoclass

        Context = autoclass("android.content.Context")  
        result: str | None = Context.getFilesDir().getParentFile().getAbsolutePath()
    except Exception:
        
        pattern = re.compile(r"/data/(data|user/\d+)/(.+)/files")
        for path in sys.path:
            if pattern.match(path):
                result = path.split("/files")[0]
                break
        else:
            result = None
    return result


@lru_cache(maxsize=1)
def _android_documents_folder() -> str:
    
    
    try:
        from jnius import autoclass

        Context = autoclass("android.content.Context")  
        Environment = autoclass("android.os.Environment")  
        documents_dir: str = Context.getExternalFilesDir(Environment.DIRECTORY_DOCUMENTS).getAbsolutePath()
    except Exception:
        documents_dir = "/storage/emulated/0/Documents"

    return documents_dir


__all__ = [
    "Android",
]
