from __future__ import annotations

import os
import sys
from configparser import ConfigParser
from pathlib import Path

from .api import PlatformDirsABC

if sys.platform.startswith("linux"):  
    from os import getuid
else:

    def getuid() -> int:
        raise RuntimeError("should only be used on Linux")


class Unix(PlatformDirsABC):
    

    @property
    def user_data_dir(self) -> str:
        
        path = os.environ.get("XDG_DATA_HOME", "")
        if not path.strip():
            path = os.path.expanduser("~/.local/share")
        return self._append_app_name_and_version(path)

    @property
    def site_data_dir(self) -> str:
        
        
        path = os.environ.get("XDG_DATA_DIRS", "")
        if not path.strip():
            path = f"/usr/local/share{os.pathsep}/usr/share"
        return self._with_multi_path(path)

    def _with_multi_path(self, path: str) -> str:
        path_list = path.split(os.pathsep)
        if not self.multipath:
            path_list = path_list[0:1]
        path_list = [self._append_app_name_and_version(os.path.expanduser(p)) for p in path_list]
        return os.pathsep.join(path_list)

    @property
    def user_config_dir(self) -> str:
        
        path = os.environ.get("XDG_CONFIG_HOME", "")
        if not path.strip():
            path = os.path.expanduser("~/.config")
        return self._append_app_name_and_version(path)

    @property
    def site_config_dir(self) -> str:
        
        
        path = os.environ.get("XDG_CONFIG_DIRS", "")
        if not path.strip():
            path = "/etc/xdg"
        return self._with_multi_path(path)

    @property
    def user_cache_dir(self) -> str:
        
        path = os.environ.get("XDG_CACHE_HOME", "")
        if not path.strip():
            path = os.path.expanduser("~/.cache")
        return self._append_app_name_and_version(path)

    @property
    def user_state_dir(self) -> str:
        
        path = os.environ.get("XDG_STATE_HOME", "")
        if not path.strip():
            path = os.path.expanduser("~/.local/state")
        return self._append_app_name_and_version(path)

    @property
    def user_log_dir(self) -> str:
        
        path = self.user_cache_dir
        if self.opinion:
            path = os.path.join(path, "log")
        return path

    @property
    def user_documents_dir(self) -> str:
        
        documents_dir = _get_user_dirs_folder("XDG_DOCUMENTS_DIR")
        if documents_dir is None:
            documents_dir = os.environ.get("XDG_DOCUMENTS_DIR", "").strip()
            if not documents_dir:
                documents_dir = os.path.expanduser("~/Documents")

        return documents_dir

    @property
    def user_runtime_dir(self) -> str:
        
        path = os.environ.get("XDG_RUNTIME_DIR", "")
        if not path.strip():
            path = f"/run/user/{getuid()}"
        return self._append_app_name_and_version(path)

    @property
    def site_data_path(self) -> Path:
        
        return self._first_item_as_path_if_multipath(self.site_data_dir)

    @property
    def site_config_path(self) -> Path:
        
        return self._first_item_as_path_if_multipath(self.site_config_dir)

    def _first_item_as_path_if_multipath(self, directory: str) -> Path:
        if self.multipath:
            
            directory = directory.split(os.pathsep)[0]
        return Path(directory)


def _get_user_dirs_folder(key: str) -> str | None:
    
    user_dirs_config_path = os.path.join(Unix().user_config_dir, "user-dirs.dirs")
    if os.path.exists(user_dirs_config_path):
        parser = ConfigParser()

        with open(user_dirs_config_path) as stream:
            
            parser.read_string(f"[top]\n{stream.read()}")

        if key not in parser["top"]:
            return None

        path = parser["top"][key].strip('"')
        
        path = path.replace("$HOME", os.path.expanduser("~"))
        return path

    return None


__all__ = [
    "Unix",
]
