"""Data parsing utilities.
"""
__author__ = 'XYZ'


import re
from math import ceil, floor

string_to_list = lambda s: [item.strip() for item in s.split(",") if item.strip()] if s else []


def normalize(s: str) -> str:
  """
  Normalizes a string by:
  - Converting to lowercase
  - Removing numeric, spaces, and special characters
  - Replacing multiple spaces with a single underscore
  - Handling None, empty strings, and invalid types gracefully
  """
  import re

  return (
    re.sub(r'\s+', '_', re.sub(r'[^a-zA-Z\s]', '', str(s or '').strip())).lower()
    if not isinstance(s, (bool, int, float, list, tuple, set)) and s is not None
    else ""
  )


## Utility functions for formatting
def escape_latex_special_characters(text):
  """
  Escapes LaTeX special characters in a string.
  Specifically handles `_`, `#`, `&`, `%`, `$`, and others.
  """
  if isinstance(text, str):
    return re.sub(r'([_#&%$])', r'\\\1', text)
  return text

def to_latex_friendly_0(text):
  """Converts a given string to LaTeX-friendly text by escaping special characters."""
  latex_special_chars = {
    '_': r'\_',
    '%': r'\%',
    '&': r'\&',
    '#': r'\#',
    '$': r'\$'
  }
  for char, replacement in latex_special_chars.items():
    text = text.replace(char, replacement)
  return text


def to_latex_friendly(text):
  """
  Converts a given string to LaTeX-friendly text by escaping special characters.
  """
  if not isinstance(text, str):
    raise ValueError("Input must be a string.")

  # Dictionary of special LaTeX characters and their escaped equivalents
  latex_special_chars = {
    '\\': r'\\textbackslash{}',
    '{': r'\{',
    '}': r'\}',
    '$': r'\$',
    '&': r'\&',
    '%': r'\%',
    '#': r'\#',
    '_': r'\_',
    '~': r'\textasciitilde{}',
    '^': r'\textasciicircum{}',
  }

  # Escape each special character in the text
  for char, replacement in latex_special_chars.items():
    text = text.replace(char, replacement)

  return text

def format_to_percentage(value):
  """Convert a value to percentage and round to 2 decimal places."""
  return f"{round(value * 100, 2):.2f}" if isinstance(value, (int, float)) else value

def format_to_millions(value):
  """Convert a value to millions and round to two decimal places."""
  return f"{round(value / 1_000_000, 2):.2f}" if isinstance(value, (int, float)) else value

def format_to_two_decimals(value):
  """Format a value to two decimal places."""
  return f"{round(value, 2):.2f}" if isinstance(value, (int, float)) else value

def format_to_ceil_integer(value):
  """Round a value up to the nearest integer."""
  return ceil(value) if isinstance(value, (int, float)) else value

def format_to_floor_integer(value):
  """Round a value down to the nearest integer."""
  return floor(value) if isinstance(value, (int, float)) else value

def format_columns(data, rules):
  """Format DataFrame columns based on the given rules."""
  for column, func_name in rules.items():
    if column in data.columns:
      # Dynamically resolve the formatting function
      func = globals().get(func_name)
      if callable(func):
        data[column] = data[column].apply(func)
      else:
        print(f"Formatting function '{func_name}' not found for column '{column}'. Skipping.")
  return data
