#!/bin/bash

##__author__ = 'XYZ'
###----------------------------------------------------------


__LSCRIPTS=$( cd "$( dirname "${BASH_SOURCE[0]}")" && pwd )

timestamp=$(date -d now +'%d%m%y_%H%M%S')


## Default configurations
BASE_DIR=${BASE_DIR:-"${__DATASET_ROOT__}/sf3d"}
OUTPUT_DIR=${OUTPUT_DIR:-"logs/sf3d"} 
BATCH_SIZE=${BATCH_SIZE:-32}
NUM_WORKERS=${NUM_WORKERS:-4}
INPUT_SIZE=${INPUT_SIZE:-"(224, 224)"}
SPLIT_RATIO=${SPLIT_RATIO:-"(0.75, 0.15, 0.1)"}
# Choose splitting method: 'by-driver' or 'by-image'
SPLIT_METHOD=${SPLIT_METHOD:-"by-driver"}

show_help() {
  echo "Usage: ./run_sf3d.sh [OPTIONS]"
  echo ""
  echo "Options:"
  echo "  --from <path>           Base dataset directory (default: ./dataset)"
  echo "  --to <path>             Output directory for saving logs (default: logs/sf3d)"
  echo "  --batch_size <int>      Batch size for DataLoader (default: 32)"
  echo "  --num_workers <int>     Number of worker threads for DataLoader (default: 4)"
  echo "  --input_size <tuple>    Input size for the DNN model (default: '(224, 224)')"
  echo "  --split_ratio <tuple>   Train/val/test split ratios (default: '(0.75, 0.15, 0.1)')"
  echo "  --split_method <str>    Splitting method: 'by-driver' or 'by-image' (default: 'by-driver')"
  echo "  -h, --help              Show this help message"
  exit 0
}

## Parse arguments
while [[ $# -gt 0 ]]; do
  case "$1" in
    --from) BASE_DIR="$2" && shift 2 ;;
    --to) OUTPUT_DIR="$2" && shift 2 ;;
    --batch_size) BATCH_SIZE="$2" && shift 2 ;;
    --num_workers) NUM_WORKERS="$2" && shift 2 ;;
    --input_size) INPUT_SIZE="$2" && shift 2 ;;
    --split_ratio) SPLIT_RATIO="$2" && shift 2 ;;
    --split_method) SPLIT_METHOD="$2" && shift 2 ;;
    -h|--help) show_help ;;
    *) echo "Unknown option: $1" && show_help ;;
  esac
done

[ -d "$BASE_DIR" ] || { echo "Error: Dataset directory '$BASE_DIR' does not exist." && exit 1; }

command -v python &> /dev/null || { echo "Error: Python is not installed or not in PATH." && exit 1; }

python -m src.datasets.sf3d \
  --from "$BASE_DIR" \
  --to "$OUTPUT_DIR" \
  --split-method "$SPLIT_METHOD" \
  --split-ratio "$SPLIT_RATIO" \
  --batch-size "$BATCH_SIZE" \
  --num-workers "$NUM_WORKERS" \
  --input-size "$INPUT_SIZE" \
  --calculate-mean-std

## Check if the script executed successfully
[ $? -eq 0 ] && echo "SF3D dataset preparation completed successfully. Output saved to: $OUTPUT_DIR" \
  || echo "Error: SF3D dataset preparation failed."
