#!/bin/bash

## __author__="XYZ"
## ----------------------------------------------------------
## Runner script for saliency visualization (panels, fg/bg, mosaics, suite)
## Works with outputs produced by: python -m src.saliency
## NOTE: This script is a thin wrapper; Python handles the logic.

set -euo pipefail

__LSCRIPTS="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
_timestamp="$(date +'%d%m%y_%H%M%S')"

## ---------------------------------------------------------------------------
## Usage
## ---------------------------------------------------------------------------
usage() {
  cat <<EOF
Usage:
  $(basename "$0") [SAL_DIR] [ARCH] [METHOD] [CORRECTNESS] [CMD] [NORM] [SCOPE] [OUT_DIR] [LIMIT] [K] [FILENAME]

Positional args (with defaults):
  1  SAL_DIR      : saliency root
                    default: \${__CODEHUB_ROOT__}/external/rflownet/logs/sage-230925_091907/saliency
  2  ARCH         : model folder under SAL_DIR (default: resnet18)
  3  METHOD       : CAM method (GradCAM | LayerCAM | EigenCAM | all) (default: EigenCAM)
  4  CORRECTNESS  : correct | incorrect | both (default: correct)
  5  CMD          : suite | panels | panel | validate | class_mosaic | overall_mosaic (default: suite)
  6  NORM         : phi_relu | minmax | zscore | pdf | none (default: phi_relu)
  7  SCOPE        : overall | per_class/cK (used by panel/panels/validate/overall_mosaic) (default: overall)
  8  OUT_DIR      : output directory (auto if empty)
  9  LIMIT        : panels/overall_mosaic row limit; 0=all (default: 0)
 10  K            : rows per class in class mosaics / suite (default: 8)
 11  FILENAME     : for panel/validate (e.g., img_34377.c7.overlay.jpg)

Examples:
  # end-to-end suite for correct/EigenCAM
  $(basename "$0") \\
    "\${__CODEHUB_ROOT__}/external/rflownet/logs/sage-230925_091907/saliency" \\
    resnet18 EigenCAM correct suite phi_relu

  # build overall mosaic for incorrect
  $(basename "$0") ... resnet18 EigenCAM incorrect overall_mosaic phi_relu overall "" 50

  # render a single panel from a group
  $(basename "$0") ... resnet18 GradCAM correct panel phi_relu overall "" 0 8 img_10350.c6.overlay.jpg

  # run full suite for ALL methods under resnet18
  $(basename "$0") ... resnet18 all correct suite
EOF
}

## ---------------------------------------------------------------------------
## Positional args + defaults
## ---------------------------------------------------------------------------
SAL_DIR="${1:-${__CODEHUB_ROOT__}/external/rflownet/logs/sage-230925_091907/saliency}"
ARCH="${2:-resnet18}"
METHOD="${3:-all}"            # or "all"
CORRECTNESS="${4:-both}"        # correct | incorrect | both
CMD="${5:-suite}"                  # suite | panels | panel | validate | class_mosaic | overall_mosaic
NORM="${6:-phi_relu}"
SCOPE="${7:-overall}"
OUT_DIR_IN="${8:-}"
LIMIT="${9:-0}"
K="${10:-8}"
FILENAME="${11:-}"

## ---------------------------------------------------------------------------
## Derived paths
## ---------------------------------------------------------------------------
ARCH_DIR="${SAL_DIR}/${ARCH}"
METHOD_DIR="${ARCH_DIR}/${METHOD}"
CORR_DIR="${METHOD_DIR}/${CORRECTNESS}"
GROUP_DIR="${CORR_DIR}/${SCOPE}"
CLASS_ROOT="${CORR_DIR}/per_class"

## ---------------------------------------------------------------------------
## Helpers
## ---------------------------------------------------------------------------
_die()  { echo "[viz:ERROR] $*" >&2; exit 1; }
_info() { echo "[viz:INFO] $*"; }
_mkdir(){ mkdir -p "$1"; }

## ---------------------------------------------------------------------------
## Sanity checks
## ---------------------------------------------------------------------------
[[ "$#" -ge 0 ]] || { usage; exit 1; }
[[ -d "${SAL_DIR}" ]]    || _die "SAL_DIR not found: ${SAL_DIR}"
[[ -d "${ARCH_DIR}" ]]   || _die "ARCH_DIR not found: ${ARCH_DIR}"

case "${CMD}" in
  suite|class_mosaic|overall_mosaic|panels|panel|validate) ;;
  -h|--help|help) usage; exit 0 ;;
  *) _die "Unknown CMD='${CMD}'. See --help for options." ;;
esac

## Default OUT dir if not provided
if [[ -z "${OUT_DIR_IN}" ]]; then
  case "${CMD}" in
    suite)
      OUT_DIR="${SAL_DIR}/${ARCH}/_viz_${_timestamp}" ;;
    class_mosaic)
      OUT_DIR="${SAL_DIR}/${ARCH}/_mosaics_${_timestamp}" ;;
    overall_mosaic|panels|panel|validate)
      OUT_DIR="${SAL_DIR}/${ARCH}/_viz_${_timestamp}" ;;
    *)
      OUT_DIR="${SAL_DIR}/${ARCH}/_viz_${_timestamp}" ;;
  esac
else
  OUT_DIR="${OUT_DIR_IN}"
fi
_mkdir "${OUT_DIR}"

## Auto-pick filename for validate/panel if not provided (first overlay)
if [[ ("${CMD}" == "validate" || "${CMD}" == "panel") && -z "${FILENAME}" ]]; then
  first_ov="$(ls "${GROUP_DIR}"/*.overlay.jpg 2>/dev/null | head -n1 || true)"
  [[ -n "${first_ov}" ]] || _die "No *.overlay.jpg in ${GROUP_DIR}; cannot auto-pick."
  FILENAME="$(basename "${first_ov}")"
fi

## ---------------------------------------------------------------------------
## Echo config
## ---------------------------------------------------------------------------
_info "SAL_DIR      : ${SAL_DIR}"
_info "ARCH         : ${ARCH}"
_info "METHOD       : ${METHOD}"
_info "CORRECTNESS  : ${CORRECTNESS}"
_info "CMD          : ${CMD}"
_info "NORM         : ${NORM}"
_info "SCOPE        : ${SCOPE}"
_info "OUT_DIR      : ${OUT_DIR}"
_info "GROUP_DIR    : ${GROUP_DIR}"
_info "CLASS_ROOT   : ${CLASS_ROOT}"
_info "LIMIT        : ${LIMIT}"
_info "K            : ${K}"
_info "FILENAME     : ${FILENAME}"

## ---------------------------------------------------------------------------
## Dispatch
## ---------------------------------------------------------------------------
case "${CMD}" in
  suite)
    python -m src.saliency_viz suite \
      --sal_root "${SAL_DIR}" \
      --arch "${ARCH}" \
      --method "${METHOD}" \
      --correctness "${CORRECTNESS}" \
      --norm "${NORM}" \
      --k "${K}" \
      --limit_overall "${LIMIT}"
    ;;

  panels)
    python -m src.saliency_viz panels \
      --group_dir "${GROUP_DIR}" \
      --method "${METHOD}" \
      --norm "${NORM}" \
      --limit "${LIMIT}" \
      --out "${OUT_DIR}"
    ;;

  panel)
    python -m src.saliency_viz panel \
      --group_dir "${GROUP_DIR}" \
      --method "${METHOD}" \
      --filename "${FILENAME}" \
      --norm "${NORM}" \
      --out "${OUT_DIR}"
    ;;

  validate)
    python -m src.saliency_viz validate \
      --group_dir "${GROUP_DIR}" \
      --method "${METHOD}" \
      --filename "${FILENAME}" \
      --norm "${NORM}" \
      --out "${OUT_DIR}"
    ;;

  class_mosaic)
    python -m src.saliency_viz class_mosaic \
      --class_root "${CLASS_ROOT}" \
      --method "${METHOD}" \
      --norm "${NORM}" \
      --k "${K}" \
      --out "${OUT_DIR}"
    ;;

  overall_mosaic)
    python -m src.saliency_viz overall_mosaic \
      --group_dir "${GROUP_DIR}" \
      --method "${METHOD}" \
      --norm "${NORM}" \
      --limit "${LIMIT}" \
      --out "${OUT_DIR}"
    ;;
esac

_info "Done."
