"""Common File Write Only utilities.
"""
__author__ = 'XYZ'


import errno
import json
import os
import re



def write_csv(to_path, df, delimiter):
  """
  Writes the parsed entries DataFrame to a CSV file using pandas, preserving the input delimiter.
  """
  os.makedirs(os.path.dirname(to_path), exist_ok=True)
  df.to_csv(to_path, sep=delimiter, index=False)
  print(f"CSV written to {to_path} with delimiter '{delimiter}'")
  print(f"Output CSV file generated: {to_path}")


def write_json(filepath, o, mode='w', indent=None, encoding='utf-8'):
  """Create a JSON file from a Python dictionary object."""
  import numpy as np

  with open(filepath, mode, encoding=encoding) as f:
    # f.write(json.dumps(o, indent=indent))
    json.dump(o, f, indent=indent, default=lambda x: x.tolist() if isinstance(x, np.ndarray) else str(x))


def mkdir_p(p):
  """mkdir -p` linux command functionality.

  References:
  * https://stackoverflow.com/questions/600268/mkdir-p-functionality-in-python
  * https://stackoverflow.com/questions/20794/find-broken-symlinks-with-python
  """
  try:
    os.makedirs(p)
  except OSError as exc:  # Python >2.5
    if exc.errno == errno.EEXIST and os.path.isdir(p):
      pass
    elif os.path.islink(p) and not os.path.exists(p):
      os.remove(p)
      mkdir_p(p)
    else:
      raise
  finally:
    return p


def prefix_normalize(s: str) -> str:
  """
  Converts the string to
  - lowercase
  - keep only alpha, numeric, minus and underscore
  """
  return (
      re.sub('[^A-Za-z_0-9-]+', '', (s or '').strip().lower())
      if not isinstance(s, bool) and
      not isinstance(s, int) and
      not isinstance(s, float) and
      not isinstance(s, list) and
      not isinstance(s, tuple) and
      not isinstance(s, set) and
      s is not None
      else s
  )


def mkdirtmp(basepath='/tmp', prefix=''):
  if not basepath:
    basepath = '/tmp'

  __ts = ts()
  p = os.path.join(basepath, __ts)
  if prefix:
    prefix = prefix_normalize(str(prefix))
    p = os.path.join(basepath, f'{prefix}{__ts}')

  mkdir_p(p)
  return p


def mkdir_basepath(basepath, prefix=''):
  __ts = ts()
  p = os.path.join(basepath, __ts)
  if prefix:
    prefix = prefix_normalize(str(prefix))
    p = os.path.join(basepath, f'{prefix}{__ts}')

  mkdir_p(p)
  return p


def mkdirts(basepath):
  return mkdirtmp(basepath=basepath)
