"""Research Flow Engine (rflow) — tiny node runner for list-based plans."""
__author__ = "XYZ"

from .core._log_ import logger
log = logger(__file__)

def _resolve_fn(step_name: str, ctx: dict):
  """
  Resolve a node function name to a callable.
  Prefers ctx['fns'] if provided; falls back to module lookup if needed.
  Accepts step_name as either 'activate_first_roots' or full 'node__activate_first_roots'.
  """
  if not step_name:
    raise ValueError("Empty step name")

  # prefer explicit node__ prefix
  fn_key = step_name if step_name.startswith("node__") else f"node__{step_name}"

  # try from ctx registry first
  fns = ctx.get("fns")
  if isinstance(fns, dict):
    fn = fns.get(fn_key)
    if fn:
      return fn

  # final fallback: look up from the main module where ctx was created
  # (caller can put the real module object in ctx['module'])
  mod = ctx.get("module")
  if mod is not None and hasattr(mod, fn_key):
    return getattr(mod, fn_key)

  raise AttributeError(f"Node function '{fn_key}' not found")

def run(plan, ctx: dict):
  """
  Execute a list-based plan:
    plan = [ {"fn": "load_image"}, {"fn": "predict"}, ... ]

  Each node receives and returns the same ctx dict.
  """
  if not isinstance(plan, (list, tuple)):
    raise TypeError("Plan must be a list/tuple of steps")

  for i, step in enumerate(plan):
    if not isinstance(step, dict) or "fn" not in step:
      raise TypeError(f"Step {i} must be a dict with key 'fn'")

    name = step["fn"]
    fn = _resolve_fn(name, ctx)

    # optional: a step may provide constant kwargs; merge-on-call
    kwargs = {k: v for k, v in step.items() if k != "fn"}
    try:
      ctx = fn(ctx, **kwargs) if kwargs else fn(ctx)
    except Exception as e:
      log.error(f"Step {i} ('{name}') failed: {e}", exc_info=True)
      # propagate error up; caller may decide how to handle
      raise

  return ctx
