#!/bin/bash

## __author__="XYZ"
## ----------------------------------------------------------
## Simple runner for: python -m src.quick_token_viz <model_dir>

set -euo pipefail

_timestamp="$(date +'%d%m%y_%H%M%S')"

usage() {
  cat <<EOF
Usage:
  $(basename "$0") MODEL_DIR [PART] [GRID] [METRIC] [UPSAMPLE] [OUT_DIR]

Arguments:
  MODEL_DIR   : required. e.g. /.../saliency/resnet18/EigenCAM
  PART        : fg | bg | whole  (default: fg)
  GRID        : 4x4 | 8x8         (default: 4x4)
  METRIC      : mean | max       (default: mean)
  UPSAMPLE    : integer (display upsample per cell) (default: 40)
  OUT_DIR     : optional output dir for snapshots (default: <MODEL_DIR>/_token_viz_<ts>)

Examples:
  ./scripts/quick_token_viz.sh /codehub/.../saliency/resnet18/EigenCAM
  ./scripts/quick_token_viz.sh /codehub/.../saliency/resnet18/EigenCAM bg 8x8 max 32 /tmp/token_viz_out
EOF
  exit 1
}

if [[ "${#}" -lt 1 ]]; then
  usage
fi

MODEL_DIR="${1}"
PART="${2:-fg}"
GRID="${3:-4x4}"
METRIC="${4:-mean}"
UPSAMPLE="${5:-40}"
OUT_DIR_IN="${6:-}"

if [[ -z "${MODEL_DIR}" || ! -d "${MODEL_DIR}" ]]; then
  echo "[qtviz:ERROR] MODEL_DIR not found or not specified: ${MODEL_DIR}" >&2
  usage
fi

if [[ -z "${OUT_DIR_IN}" ]]; then
  OUT_DIR="${MODEL_DIR}/_token_viz_${_timestamp}"
else
  OUT_DIR="${OUT_DIR_IN}"
fi
mkdir -p "${OUT_DIR}"

echo "[qtviz:INFO] model_dir : ${MODEL_DIR}"
echo "[qtviz:INFO] part      : ${PART}"
echo "[qtviz:INFO] grid      : ${GRID}"
echo "[qtviz:INFO] metric    : ${METRIC}"
echo "[qtviz:INFO] upsample  : ${UPSAMPLE}"
echo "[qtviz:INFO] out_dir   : ${OUT_DIR}"

# run the viz module
python -m src.quick_token_viz "${MODEL_DIR}" \
  --part "${PART}" \
  --grid "${GRID}" \
  --metric "${METRIC}" \
  --upsample "${UPSAMPLE}" \
  --out "${OUT_DIR}"
rc=$?

if [[ ${rc} -ne 0 ]]; then
  echo "[qtviz:ERROR] viz exited with code ${rc}" >&2
  exit ${rc}
fi

echo "[qtviz:INFO] Done. Snapshots (if any) saved to: ${OUT_DIR}"
