"""
Utility module to capture system details, GPU information, and record timing stats.
"""
__author__ = 'XYZ'


import getpass
import grp
import os
import platform
import socket
import time

from datetime import datetime


import pandas as pd

try:
  import torch
except ImportError:
  print('torch is not installed')


def get_system_info():
  """Retrieve detailed system information."""
  return {
    "hostname": socket.gethostname(),
    "platform": platform.system(),
    "platform_version": platform.version(),
    "architecture": platform.architecture()[0],
    "processor": platform.processor(),
    "username": getpass.getuser(),
    "uid": os.getuid(),
    "gid": os.getgid(),
    "groupname": grp.getgrgid(os.getgid()).gr_name,
  }

def get_gpu_info():
  """Retrieve GPU information."""
  try:
    if torch.cuda.is_available():
      gpu_name = torch.cuda.get_device_name(0)
      gpu_memory = torch.cuda.get_device_properties(0).total_memory / (1024 ** 3)  # Convert bytes to GB
      return {
        "gpu_name": gpu_name,
        "gpu_memory_gb": round(gpu_memory, 2)
      }
    else:
      return {"gpu_name": "No GPU available", "gpu_memory_gb": 0}
  except Exception as e:
    return {"gpu_name": f"Error retrieving GPU info: {e}", "gpu_memory_gb": 0}

def save_system_info(to_path):
  """Save system and GPU information to a CSV file."""
  system_info = get_system_info()
  gpu_info = get_gpu_info()
  system_info.update(gpu_info)

  file_path = os.path.join(to_path, "system_info.csv")
  df = pd.DataFrame([system_info])
  df.to_csv(file_path, index=False)
  print(f"System info saved to: {file_path}")

def save_timings(to_path, training_start, training_end, eval_start, eval_end):
  """Save training and evaluation timing stats to a CSV file."""
  file_path = os.path.join(to_path, "timings.csv")
  headers = ["stage", "start_time", "end_time", "duration_minutes"]
  timing_data = []

  ## Calculate durations
  training_duration = (training_end - training_start) / 60
  eval_duration = (eval_end - eval_start) / 60

  ## Add timing stats
  timing_data.append(["train", training_start, training_end, training_duration])
  timing_data.append(["eval", eval_start, eval_end, eval_duration])

  ## Save to CSV
  df = pd.DataFrame(timing_data, columns=headers)
  df.to_csv(file_path, index=False)
  print(f"Timing stats saved to: {file_path}")

if __name__ == "__main__":
  ## Test the module standalone
  to_path = os.path.join('logs', f"sysutils-{datetime.now().strftime('%d%m%y_%H%M%S')}")
  os.makedirs(to_path, exist_ok=True)

  ## Test system info
  save_system_info(to_path)

  ## Test timings
  start_train = time.time()
  time.sleep(1)  ## Simulate training
  end_train = time.time()

  start_eval = time.time()
  time.sleep(0.5)  ## Simulate evaluation
  end_eval = time.time()

  save_timings(to_path, start_train, end_train, start_eval, end_eval)
