#!/bin/bash

## __author__="XYZ"
## ----------------------------------------------------------
## Loop runner for python -m src.quick_token_check over saliency basepath

set -euo pipefail

_timestamp="$(date +'%d%m%y_%H%M%S')"

usage() {
  cat <<EOF
Usage:
  $(basename "$0") SAL_BASE [ARCH_FILTER] [METHOD_FILTER]

Arguments:
  SAL_BASE       : root saliency directory (required)
  ARCH_FILTER    : optional glob or regex substring to filter arch folders (default: all)
  METHOD_FILTER  : optional glob or regex substring to filter method folders (default: all)

Example:
  $(basename "$0") /codehub/external/rflownet/logs/sage-230925_091907/saliency
  $(basename "$0") /path/to/saliency resnet18 EigenCAM
EOF
}

if [[ "${#}" -lt 1 ]]; then
  usage
  exit 1
fi

SAL_BASE="${1}"
ARCH_FILTER="${2:-}"
METHOD_FILTER="${3:-}"

_die()  { echo "[qt-loop:ERROR] $*" >&2; exit 2; }
_info() { echo "[qt-loop:INFO] $*"; }

[[ -d "${SAL_BASE}" ]] || _die "SAL_BASE not found: ${SAL_BASE}"

# Where the python module lives — adjust if your layout differs
PY_MODULE="src.quick_token_check"

_info "SAL_BASE   : ${SAL_BASE}"
_info "ARCH_FILTER: ${ARCH_FILTER}"
_info "METHOD_FILTER: ${METHOD_FILTER}"

# iterate arch folders
for arch_path in "${SAL_BASE}"/*; do
  [[ -d "${arch_path}" ]] || continue
  arch_name="$(basename "${arch_path}")"
  if [[ -n "${ARCH_FILTER}" && "${arch_name}" != *"${ARCH_FILTER}"* ]]; then
    _info "Skipping arch (filter): ${arch_name}"
    continue
  fi

  # iterate method folders under arch
  for method_path in "${arch_path}"/*; do
    [[ -d "${method_path}" ]] || continue
    method_name="$(basename "${method_path}")"
    if [[ -n "${METHOD_FILTER}" && "${method_name}" != *"${METHOD_FILTER}"* ]]; then
      _info "Skipping method (filter): ${arch_name}/${method_name}"
      continue
    fi

    # check if tokens exist under this path before running
    token_check_count=$(find "${method_path}" -type f -name "*.tokens.npy" | head -n1 || true)
    if [[ -z "${token_check_count}" ]]; then
      _info "No tokens found under: ${arch_name}/${method_name} — skipping."
      continue
    fi

    _info "Running quick_token_check on: ${arch_name}/${method_name}"
    set +e
    python -m "${PY_MODULE}" "${method_path}"
    rc=$?
    set -e
    if [[ ${rc} -ne 0 ]]; then
      echo "[qt-loop:WARN] quick_token_check failed for ${arch_name}/${method_name} (rc=${rc})"
    else
      _info "Completed quick_token_check for ${arch_name}/${method_name}"
    fi
  done
done

_info "Done. Loop finished at ${_timestamp}"
