"""Logging configuration.

References:
* https://docs.python.org/3/howto/logging-cookbook.html#an-example-dictionary-based-configuration
* https://docs.djangoproject.com/en/1.9/topics/logging/#configuring-logging
"""
__author__ = 'XYZ'
__all__ = [
    'log',
]


import logging
import logging.config
import os
from typing import Dict, List, Optional

from . import color


class Logcfg:
  """Logging configuration for python logging module."""

  @staticmethod
  def get_log_levels():
    return [
        'CRITICAL',
        'ERROR',
        'WARNING',
        'INFO',
        'DEBUG',
        'NOTSET',
    ]

  cfg: Dict = {
      'version': 1,
      'disable_existing_loggers': False,
      'formatters': {
          'verbose': {
              'format': '%(levelname)s %(asctime)s %(module)s %(process)d %(thread)d %(message)s',
          },
          'simple': {
              # 'format': '%(levelname)s %(message)s'
              'format': '[%(levelname)s]:[%(filename)s:%(lineno)d - %(funcName)20s()]: %(message)s',
              # 'format': '%(asctime)s - %(name)s - %(levelname)s - %(message)s'
          },
          'standard': {
              # 'format': '%(asctime)s:[%(levelname)s]:%(lineno)d:%(name)s:: %(message)s'
              # 'format': '[%(asctime)s:[%(levelname)s]:[%(name)s]:[%(filename)s:%(lineno)d - %(funcName)20s()]: %(message)s'
              'format': '[%(asctime)s:[%(levelname)s]:[%(filename)s:%(lineno)d - %(funcName)20s()]: %(message)s',
          },
          # 'json': {
          #   'format': {
          #     'time':'%(asctime)s',
          #     'level':'%(levelname)s',
          #     'logger':'%(name)s',
          #     'filename':'%(filename)s',
          #     'lineno':'%(lineno)d',
          #     'funcName':'%(funcName)20s',
          #     'message':'%(message)s',
          #   }
          # },
      },
      'filters': {},
      'handlers': {
          'default': {
              'level': 'DEBUG',
              'class': 'logging.StreamHandler',
          },
          'console': {
              'level': 'DEBUG',
              'class': 'logging.StreamHandler',
              'formatter': 'standard',
              'stream': 'ext://sys.stdout',
          },
          # 'file_info': {
          #   'class': 'logging.handlers.RotatingFileHandler',
          #   'level': 'DEBUG',
          #   'formatter': 'standard',
          #   'filename': '.genesis.log',
          #   'maxBytes': 10485760,
          #   # 'maxBytes': 10,
          #   # 'backupCount': 20,
          #   'encoding': 'utf8'
          # },
          # 'file_error': {
          #   'class': 'logging.handlers.RotatingFileHandler',
          #   'level': 'ERROR',
          #   'formatter': 'standard',
          #   'filename': 'log/errors.log',
          #   'maxBytes': 10485760,
          #   'backupCount': 20,
          #   'encoding': 'utf8'
          # }
      },
      'loggers': {
          '': {'handlers': ['console'], 'level': 'INFO', 'propagate': False},
          '__main__': {
              'handlers': ['console'],
              # 'handlers': ['file_info'],
              # 'level': 'CRITICAL',
              # 'level': 'ERROR',
              # 'level': 'WARNING',
              # 'level': 'INFO',
              # 'level': 'DEBUG',
              # 'level': 'NOTSET',
              'propagate': False,
          },
      },
  }

  level: str = 'DEBUG'
  handler: List = ['console']
  verbose: str = '%(levelname)s %(asctime)s %(module)s %(process)d %(thread)d %(message)s'
  simple: str = '[%(levelname)s]:[%(filename)s:%(lineno)d - %(funcName)20s()]: %(message)s'
  standard: str = '[%(levelname)s - %(asctime)s]:[%(filename)s:%(lineno)d - %(funcName)20s()]: %(message)s'
  ccstandard: str = '[{}%(levelname)s{} - %(asctime)s]:{}[%(filename)s:%(lineno)d - %(funcName)20s()]{}: %(message)s{}'

  def __init__(self, **kwargs):
    self.level = kwargs.get('level', 'DEBUG')
    # set level
    self.cfg['loggers']['__main__']['level'] = self.level
    fn = getattr(self, self.level.lower())
    self.cfg['formatters']['standard']['format'] = fn()

  def critical(self):
    return self.ccstandard.format(
        color.on_ired,
        color.on_ired,
        color.bwhi,
        color.on_ired,
        color.nocolor,
    )

  def error(self):
    return self.ccstandard.format(
        color.bired,
        color.bired,
        color.bwhi,
        color.bired,
        color.nocolor,
    )

  def warning(self):
    return self.ccstandard.format(
        color.biyel,
        color.biyel,
        color.bwhi,
        color.biyel,
        color.nocolor,
    )

  def info(self):
    return self.ccstandard.format(
        color.bblu,
        color.bblu,
        color.bwhi,
        color.bblu,
        color.nocolor,
    )

  def debug(self):
    return self.ccstandard.format(
        color.bired,
        color.bcya,
        color.bwhi,
        color.bcya,
        color.nocolor,
    )


def get_logcfg():
  _SKY__LOG_LEVEL = os.getenv('_SKY__LOG_LEVEL')

  LOG_LEVELS = Logcfg.get_log_levels()
  _SKY__LOG_LEVEL = (
      _SKY__LOG_LEVEL.upper()
      if _SKY__LOG_LEVEL and _SKY__LOG_LEVEL in LOG_LEVELS
      else 'ERROR'
  )
  ## color.cprint('Current Logger Level [_SKY__LOG_LEVEL] is: {}'.format(_SKY__LOG_LEVEL))
  kwargs = {'level': _SKY__LOG_LEVEL}
  logcfg = Logcfg(**kwargs)
  return logcfg.cfg


def supress_warnings():
  """Supress pesky warnings explicitly. Always enable these warnings in debug
  logging mode, i.e. _SKY__LOG_LEVEL=='DEBUG' overrides _SKY__FILTERWARNINGS
  flag.

  References:
  1. How to suppress “Future warning” tensorflow?
    * https://stackoverflow.com/a/64857211
  """
  import warnings

  _SKY__LOG_LEVEL = os.getenv('_SKY__LOG_LEVEL')
  _SKY__FILTERWARNINGS = os.getenv('_SKY__FILTERWARNINGS')

  if _SKY__LOG_LEVEL == 'DEBUG' or _SKY__FILTERWARNINGS:
    warnings.filterwarnings(
        'ignore',
        message='numpy.dtype size changed',
    )  # noqa: E402
    warnings.filterwarnings(
        'ignore',
        message='numpy.ufunc size changed',
    )  # noqa: E402
    warnings.filterwarnings(
        'ignore',
        message=r'Passing',
        category=FutureWarning,
    )  # noqa: E402

def logger(module_name: str, log_dir: Optional[str] = None, json_format: bool = False, has_console: bool = True, has_file: bool = False):
  logcfg = get_logcfg()
  log = logging.getLogger('__main__.' + __name__)
  logging.config.dictConfig(logcfg)

  # log.critical('Hello')
  # log.error('Hello')
  # log.warning('Hello')
  # log.info('Hello')
  # log.debug('Hello')

  return log
