import argparse
import torch
torch.manual_seed(3407)
import numpy as np 

from copy import deepcopy
from sklearn.model_selection import train_test_split

from utils import get_dataset_from_OPENML, preprocess_df
from config import NOMINIAL_COLS, NUMERICAL_COLS, UNL_FEATURE_NUM
from config import get_ori_model_save_path
from config import get_BL2_model_save_path, get_BL2_model_acc_save_path, get_BL2_model_y_pred_save_path, get_BL2_model_running_time_save_path, get_BL2_model_train_loss_save_path
from utils import TabularClassifier
from utils import BL2_prep_data, train_BL2_model, evaluate_classifier


if __name__ == "__main__":
    parser = argparse.ArgumentParser() 
    
    parser.add_argument('--dataset_name', default='EYE_MOVEMENTS', type=str, help='the name of the dataset')
    parser.add_argument('--device', default='cuda', type=str, help='device')
    parser.add_argument('--ori_training_epochs', default=1500, type=int, help='original model training epochs')    
    parser.add_argument('--BL2_unlearning_epochs', default=2, type=int, help='unlearning epochs for baseline 2')
    parser.add_argument('--unlearn_times', default=3, type=int, help='unlearn times')
    
    args = parser.parse_args()
    
    dataset_name = args.dataset_name
    device = torch.device(args.device if torch.cuda.is_available() else 'cpu')
    ori_training_epochs = args.ori_training_epochs
    BL2_unlearning_epochs = args.BL2_unlearning_epochs
    unlearn_times = args.unlearn_times
    unl_feature_num = UNL_FEATURE_NUM[dataset_name]
    
    
    # get data 
    df = get_dataset_from_OPENML(dataset_name=dataset_name)
    output_dim = len(df[df.columns[-1]].unique())
    
    # split data
    nominial_cols, numerical_cols = NOMINIAL_COLS[dataset_name], NUMERICAL_COLS[dataset_name]
    X_processed, y = preprocess_df(df=df, numerical_cols=numerical_cols, nominial_cols=nominial_cols)
    if dataset_name == 'COMPASS': X_processed = X_processed.toarray()
    # split dataset for original training
    X_train, X_test, y_train, y_test = train_test_split(X_processed, y, test_size=0.2, random_state=42)
    # split dataset for retraining from scratch
    RT_X_processed = deepcopy(X_processed[:, unl_feature_num:])
    RT_X_train, RT_X_test, _, _ = train_test_split(RT_X_processed, y, test_size=0.2, random_state=42)
    
    
    
    
    
    # load original model
    ori_input_dim = X_train.shape[1]
    ori_model = TabularClassifier(input_dim=ori_input_dim, output_dim=output_dim).to(device)
    ori_model.load_state_dict(torch.load(get_ori_model_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs), map_location=device))
    
    
    # prep data 
    BL2_model_lst, BL2_training_time_lst, BL2_train_loss_lst = [], [], []
    BL2_acc_lst, BL2_y_pred_lst = [], []
    for _ in range(unlearn_times):
        BL2_model, BL2_X_train = BL2_prep_data(X_train=X_train, model=ori_model, unl_feature_num=unl_feature_num)
        # baseline 2
        BL2_model, BL2_training_time, BL2_train_loss = train_BL2_model(BL2_X_train=BL2_X_train, y_train=y_train, X_train=X_train, BL2_model=BL2_model, epochs=BL2_unlearning_epochs, device=device)
        # evaluate baseline 2
        BL2_acc, BL2_y_pred, _ = evaluate_classifier(model=BL2_model, X_test=X_test, y_test=y_test, device=device)
        
        BL2_model_lst.append(BL2_model)
        BL2_training_time_lst.append(BL2_training_time)
        BL2_train_loss_lst.append(BL2_train_loss)
        BL2_acc_lst.append(BL2_acc)
        BL2_y_pred_lst.append(BL2_y_pred)



    
    # save res
    np.savetxt(get_BL2_model_acc_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs, BL2_unlearning_epochs=BL2_unlearning_epochs), [np.mean(BL2_acc_lst), np.std(BL2_acc_lst)])
    np.savetxt(get_BL2_model_running_time_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs, BL2_unlearning_epochs=BL2_unlearning_epochs), [np.mean(BL2_training_time_lst), np.std(BL2_training_time_lst)])    
    
    # save model 
    for unlearn_times_idx, (BL2_model, BL2_y_pred, BL2_train_loss) in enumerate(zip(BL2_model_lst, BL2_y_pred_lst, BL2_train_loss_lst)):
        
        torch.save(BL2_model.state_dict(), get_BL2_model_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs, BL2_unlearning_epochs=BL2_unlearning_epochs, unlearn_times_idx=unlearn_times_idx))
        np.savetxt(get_BL2_model_y_pred_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs, BL2_unlearning_epochs=BL2_unlearning_epochs, unlearn_times_idx=unlearn_times_idx), BL2_y_pred)
        np.savetxt(get_BL2_model_train_loss_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs, BL2_unlearning_epochs=BL2_unlearning_epochs, unlearn_times_idx=unlearn_times_idx), BL2_train_loss)
    

