import argparse
import torch
torch.manual_seed(3407)
import numpy as np 

from copy import deepcopy
from sklearn.model_selection import train_test_split

from utils import get_dataset_from_OPENML, preprocess_df
from config import NOMINIAL_COLS, NUMERICAL_COLS, UNL_FEATURE_NUM
from config import get_RT_model_save_path, get_RT_model_acc_save_path, get_RT_model_y_pred_save_path, get_RT_model_running_time_save_path, get_RT_model_train_loss_save_path
from utils import train_classifier, evaluate_classifier

if __name__ == "__main__":
    parser = argparse.ArgumentParser() 
    
    parser.add_argument('--dataset_name', default='EYE_MOVEMENTS', type=str, help='the name of the dataset')
    parser.add_argument('--device', default='cuda', type=str, help='device')
    parser.add_argument('--RT_training_epochs', default=1500, type=int, help='model training epochs of retrained model')    
    parser.add_argument('--RT_train_times', default=3, type=int, help='how many times to retrain the model -- take the avg acc')
    
    args = parser.parse_args()
    
    dataset_name = args.dataset_name
    device = torch.device(args.device if torch.cuda.is_available() else 'cpu')
    RT_training_epochs = args.RT_training_epochs
    RT_train_times = args.RT_train_times
    unl_feature_num = UNL_FEATURE_NUM[dataset_name] 
    
    # get data 
    df = get_dataset_from_OPENML(dataset_name=dataset_name)
    output_dim = len(df[df.columns[-1]].unique())
    
    # split data
    nominial_cols, numerical_cols = NOMINIAL_COLS[dataset_name], NUMERICAL_COLS[dataset_name]
    X_processed, y = preprocess_df(df=df, numerical_cols=numerical_cols, nominial_cols=nominial_cols)
    if dataset_name == 'COMPASS': X_processed = X_processed.toarray()
    # split dataset for original training
    X_train, X_test, y_train, y_test = train_test_split(X_processed, y, test_size=0.2, random_state=42)
    # split dataset for retraining from scratch
    RT_X_processed = deepcopy(X_processed[:, unl_feature_num:])
    RT_X_train, RT_X_test, _, _ = train_test_split(RT_X_processed, y, test_size=0.2, random_state=42)

    RT_model_lst, RT_training_time_lst, RT_train_loss_lst = [], [], []
    RT_acc_lst, RT_y_pred_lst = [], []
    for _ in range(RT_train_times):
        RT_model, RT_training_time, RT_train_loss = train_classifier(X_train=RT_X_train, y_train=y_train, output_dim=output_dim, device=device, epochs=RT_training_epochs)
        RT_acc, RT_y_pred, _ = evaluate_classifier(model=RT_model, X_test=RT_X_test, y_test=y_test, device=device)
        
        RT_model_lst.append(RT_model)
        RT_training_time_lst.append(RT_training_time)
        RT_train_loss_lst.append(RT_train_loss)
        RT_acc_lst.append(RT_acc)
        RT_y_pred_lst.append(RT_y_pred)
    
    
    # save res 
    np.savetxt(get_RT_model_acc_save_path(dataset_name=dataset_name, RT_training_epochs=RT_training_epochs), [np.mean(RT_acc_lst), np.std(RT_acc_lst)])
    np.savetxt(get_RT_model_running_time_save_path(dataset_name=dataset_name, RT_training_epochs=RT_training_epochs), [np.mean(RT_training_time_lst), np.std(RT_training_time_lst)])
    
    for train_times_idx, (RT_model, RT_y_pred, RT_train_loss) in enumerate(zip(RT_model_lst, RT_y_pred_lst, RT_train_loss_lst)):        
        torch.save(RT_model.state_dict(), get_RT_model_save_path(dataset_name=dataset_name, RT_training_epochs=RT_training_epochs, RT_train_times_idx=train_times_idx))
        np.savetxt(get_RT_model_y_pred_save_path(dataset_name=dataset_name, RT_training_epochs=RT_training_epochs, RT_train_times_idx=train_times_idx), RT_y_pred)
        np.savetxt(get_RT_model_train_loss_save_path(dataset_name=dataset_name, RT_training_epochs=RT_training_epochs, RT_train_times_idx=train_times_idx), RT_train_loss)
    

    