import argparse
import torch
torch.manual_seed(3407)
import numpy as np 

from copy import deepcopy
from sklearn.model_selection import train_test_split

from utils import get_dataset_from_OPENML, preprocess_df
from config import NOMINIAL_COLS, NUMERICAL_COLS
from config import get_ori_model_save_path, get_ori_model_acc_save_path, get_ori_model_y_pred_save_path, get_ori_model_running_time_save_path, get_ori_model_train_loss_save_path
from utils import train_classifier, evaluate_classifier



if __name__ == "__main__":
    parser = argparse.ArgumentParser() 
    
    parser.add_argument('--dataset_name', default='ELECTRICITY', type=str, help='the name of the dataset')
    parser.add_argument('--device', default='cuda', type=str, help='device')
    parser.add_argument('--ori_training_epochs', default=2, type=int, help='original model training epochs')    
    
    args = parser.parse_args()
    
    dataset_name = args.dataset_name
    device = torch.device(args.device if torch.cuda.is_available() else 'cpu')
    ori_training_epochs = args.ori_training_epochs
    
    print(device)
    
    # get data 
    df = get_dataset_from_OPENML(dataset_name=dataset_name)
    
    print(df)
    output_dim = len(df[df.columns[-1]].unique())
    
    print(output_dim)
    
    # split data
    nominial_cols, numerical_cols = NOMINIAL_COLS[dataset_name], NUMERICAL_COLS[dataset_name]
    X_processed, y = preprocess_df(df=df, numerical_cols=numerical_cols, nominial_cols=nominial_cols)
    # split dataset for original training
    X_train, X_test, y_train, y_test = train_test_split(X_processed, y, test_size=0.2, random_state=42)
    # split dataset for retraining from scratch
    RT_X_processed = deepcopy(X_processed[:, 1:])
    RT_X_train, RT_X_test, _, _ = train_test_split(RT_X_processed, y, test_size=0.2, random_state=42)
    
    
    
    # train original model 
    ori_model, ori_training_time, ori_train_loss_lst = train_classifier(
        X_train=X_train, y_train=y_train, 
        output_dim=output_dim, device=device, epochs=ori_training_epochs
    )
    # evaluate original model
    ori_acc, ori_y_pred, _ = evaluate_classifier(model=ori_model, X_test=X_test, y_test=y_test, device=device)    
    
    # save model
    torch.save(ori_model.state_dict(), get_ori_model_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs))
    
    # save res 
    np.savetxt(get_ori_model_acc_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs), [ori_acc])
    
    # save predictions
    np.savetxt(get_ori_model_y_pred_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs), ori_y_pred)
    
    # save running time    
    np.savetxt(get_ori_model_running_time_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs), [ori_training_time])
    
    # save training loss 
    np.savetxt(get_ori_model_train_loss_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs), ori_train_loss_lst)
    
    
    
    
    
    
    
    
    
    
    