import argparse
import torch
torch.manual_seed(3407)
import numpy as np 

from copy import deepcopy
from sklearn.model_selection import train_test_split

from utils import get_dataset_from_OPENML, preprocess_df
from config import NOMINIAL_COLS, NUMERICAL_COLS
from config import get_ori_model_save_path
from config import get_our_model_save_path, get_our_model_acc_save_path, get_our_model_y_pred_save_path, get_our_model_running_time_save_path, get_our_model_train_loss_save_path
from utils import TabularClassifier
from utils import train_unlearning_with_shuffle
from utils import evaluate_classifier




if __name__ == "__main__":
    parser = argparse.ArgumentParser() 
    
    parser.add_argument('--dataset_name', default='RL', type=str, help='the name of the dataset')
    parser.add_argument('--device', default='cuda', type=str, help='device')
    parser.add_argument('--ori_training_epochs', default=1500, type=int, help='original model training epochs')    
    parser.add_argument('--unlearning_epochs', default=200, type=int, help='our unlearning epochs')
    parser.add_argument('--unlearn_times', default=10, type=int, help='unlearn times')
    
    args = parser.parse_args()
    
    dataset_name = args.dataset_name
    device = torch.device(args.device if torch.cuda.is_available() else 'cpu')
    ori_training_epochs = args.ori_training_epochs
    unlearning_epochs = args.unlearning_epochs
    unlearn_times = args.unlearn_times
    
    
    # get data 
    df = get_dataset_from_OPENML(dataset_name=dataset_name)
    output_dim = len(df[df.columns[-1]].unique())
    
    # split data
    nominial_cols, numerical_cols = NOMINIAL_COLS[dataset_name], NUMERICAL_COLS[dataset_name]
    X_processed, y = preprocess_df(df=df, numerical_cols=numerical_cols, nominial_cols=nominial_cols)
    # split dataset for original training
    X_train, X_test, y_train, y_test = train_test_split(X_processed, y, test_size=0.2, random_state=42)
    # split dataset for retraining from scratch
    RT_X_processed = deepcopy(X_processed[:, 1:])
    RT_X_train, RT_X_test, _, _ = train_test_split(RT_X_processed, y, test_size=0.2, random_state=42)
    
    
    
    
    # load original model
    UL_model_lst, UL_training_time_lst, UL_train_loss_lst = [], [], []
    UL_acc_lst, UL_y_pred_lst = [], []
    for _ in range(unlearn_times):
        ori_input_dim = X_train.shape[1]
        UL_model = TabularClassifier(input_dim=ori_input_dim, output_dim=output_dim).to(device)
        UL_model.load_state_dict(torch.load(get_ori_model_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs), map_location=device))    
        
        
        # unlearn original model
        UL_model, task_loss_lst, UL_training_time = train_unlearning_with_shuffle(UL_model=UL_model, X_train=X_train, y_train=y_train, device=device, epochs=unlearning_epochs)
        # evaluate unlearned model 
        UL_acc, UL_y_pred, _ = evaluate_classifier(model=UL_model, X_test=X_test, y_test=y_test, device=device)
        
        
        UL_model_lst.append(UL_model)
        UL_training_time_lst.append(UL_training_time)
        UL_train_loss_lst.append(task_loss_lst)
        UL_acc_lst.append(UL_acc)
        UL_y_pred_lst.append(UL_y_pred)
        
        
    
    # save res
    np.savetxt(get_our_model_acc_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs, unlearning_epochs=unlearning_epochs), [np.mean(UL_acc_lst), np.std(UL_acc_lst)])
    np.savetxt(get_our_model_running_time_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs, unlearning_epochs=unlearning_epochs), [np.mean(UL_training_time_lst), np.std(UL_training_time_lst)])

    # save model
    for unlearn_times_idx, (UL_model, UL_y_pred, UL_train_loss) in enumerate(zip(UL_model_lst, UL_y_pred_lst, UL_train_loss_lst)):
        torch.save(UL_model.state_dict(), get_our_model_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs, unlearning_epochs=unlearning_epochs, unlearn_times_idx=unlearn_times_idx))
        np.savetxt(get_our_model_y_pred_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs, unlearning_epochs=unlearning_epochs, unlearn_times_idx=unlearn_times_idx), UL_y_pred)
        np.savetxt(get_our_model_train_loss_save_path(dataset_name=dataset_name, ori_training_epochs=ori_training_epochs, unlearning_epochs=unlearning_epochs, unlearn_times_idx=unlearn_times_idx), UL_train_loss)
    
    
    