import argparse
import torch
torch.manual_seed(3407)
import numpy as np 

from copy import deepcopy
from sklearn.model_selection import train_test_split

from utils import get_dataset_from_OPENML, preprocess_df
from config import NOMINIAL_COLS, NUMERICAL_COLS
from config import get_ori_BL3RepDetExtractormodel_save_path, get_ori_BL3Classifier_model_save_path
from config import get_BL3DecoderMIhx_model_save_path, get_BL3MIhy_model_save_path, get_BL3MIhz_model_save_path, get_BL3_prep_running_time_save_path
from utils import BL3RepDetExtractor, BL3Classifier
from utils import get_h_for_BL3
from utils import train_BL3DecoderMIhx, train_BL3MIhy, train_BL3MIhz

if __name__ == "__main__":
    parser = argparse.ArgumentParser() 
    
    parser.add_argument('--dataset_name', default='ELECTRICITY', type=str, help='the name of the dataset')
    parser.add_argument('--device', default='cuda', type=str, help='device')
    parser.add_argument('--ori_BL3_training_epochs', default=2, type=int, help='original model training epochs')    
    parser.add_argument('--BL3DecoderMIhx_training_epochs', default=2, type=int, help='training epochs of baseline 3 decoder for I(h,x), x is remaining features')
    parser.add_argument('--BL3MIhy_training_epochs', default=2, type=int, help='training epochs of baseline 3 for I(h,y), y is label')
    parser.add_argument('--BL3MIhz_training_epochs', default=2, type=int, help='training epochs of baseline 3 for I(h,z), z is unlearned feature')
    
    args = parser.parse_args()
    
    dataset_name = args.dataset_name
    device = torch.device(args.device if torch.cuda.is_available() else 'cpu')
    ori_BL3_training_epochs = args.ori_BL3_training_epochs
    BL3DecoderMIhx_training_epochs = args.BL3DecoderMIhx_training_epochs
    BL3MIhy_training_epochs = args.BL3MIhy_training_epochs
    BL3MIhz_training_epochs = args.BL3MIhz_training_epochs
    
    # get data 
    df = get_dataset_from_OPENML(dataset_name=dataset_name)
    output_dim = len(df[df.columns[-1]].unique())
    
    # split data
    nominial_cols, numerical_cols = NOMINIAL_COLS[dataset_name], NUMERICAL_COLS[dataset_name]
    X_processed, y = preprocess_df(df=df, numerical_cols=numerical_cols, nominial_cols=nominial_cols)
    # split dataset for original training
    X_train, X_test, y_train, y_test = train_test_split(X_processed, y, test_size=0.2, random_state=42)
    # split dataset for retraining from scratch
    RT_X_processed = deepcopy(X_processed[:, 1:])
    RT_X_train, RT_X_test, _, _ = train_test_split(RT_X_processed, y, test_size=0.2, random_state=42)
    
    
    
    
    
    # load original model
    BL3RepDetExtractor_model = BL3RepDetExtractor(input_dim=X_train.shape[1]).to(device)
    BL3Classifier_model = BL3Classifier(output_dim=output_dim).to(device)
    BL3RepDetExtractor_model.load_state_dict(torch.load(get_ori_BL3RepDetExtractormodel_save_path(dataset_name=dataset_name, ori_BL3_training_epochs=ori_BL3_training_epochs), map_location=device))
    BL3Classifier_model.load_state_dict(torch.load(get_ori_BL3Classifier_model_save_path(dataset_name=dataset_name, ori_BL3_training_epochs=ori_BL3_training_epochs), map_location=device))
    
    # get h     
    h_train = get_h_for_BL3(X_train=X_train, device=device, BL3RepDetExtractor_model=BL3RepDetExtractor_model)
    
    # train BL3DecoderMIhx
    BL3DecoderMIhx_model, BL3DecoderMIhx_training_time = train_BL3DecoderMIhx(h_train=h_train, X_train=X_train, device=device, epochs=BL3DecoderMIhx_training_epochs, output_dim=X_train.shape[1]-1)
    # train BL3MIhy
    BL3MIhy_model, BL3MIhy_training_time = train_BL3MIhy(h_train=h_train, y_train=y_train, device=device, epochs=BL3MIhy_training_epochs, output_dim=output_dim)
    # train BL3MIhz
    BL3MIhz_model, BL3MIhz_training_time = train_BL3MIhz(h_train=h_train, X_train=X_train, device=device, epochs=BL3MIhz_training_epochs)
    
    overall_training_time = BL3DecoderMIhx_training_time + BL3MIhy_training_time + BL3MIhz_training_time
    

    torch.save(BL3DecoderMIhx_model.state_dict(), get_BL3DecoderMIhx_model_save_path(dataset_name=dataset_name, ori_BL3_training_epochs=ori_BL3_training_epochs, BL3DecoderMIhx_training_epochs=BL3DecoderMIhx_training_epochs))
    torch.save(BL3MIhy_model.state_dict(), get_BL3MIhy_model_save_path(dataset_name=dataset_name, ori_BL3_training_epochs=ori_BL3_training_epochs, BL3MIhy_training_epochs=BL3MIhy_training_epochs))
    torch.save(BL3MIhz_model.state_dict(), get_BL3MIhz_model_save_path(dataset_name=dataset_name, ori_BL3_training_epochs=ori_BL3_training_epochs, BL3MIhz_training_epochs=BL3MIhz_training_epochs))
    
    
    np.savetxt(get_BL3_prep_running_time_save_path(dataset_name=dataset_name, ori_BL3_training_epochs=ori_BL3_training_epochs, BL3DecoderMIhx_training_epochs=BL3DecoderMIhx_training_epochs, BL3MIhy_training_epochs=BL3MIhy_training_epochs, BL3MIhz_training_epochs=BL3MIhz_training_epochs), [overall_training_time])
