package x.y.z;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import de.unima.ki.anyburl.data.Triple;
import de.unima.ki.anyburl.data.TripleSet;
import de.unima.ki.anyburl.eval.CompletionResult;
import de.unima.ki.anyburl.eval.HitsAtK;
import de.unima.ki.anyburl.eval.ResultSet;



public class Rescorer {
		
	private TripleSet train; 
	private TripleSet valid;
	
	private static final int TOP_K = 100;
	
	
	private LinkedList<String> relations = new LinkedList<String>();
	
	private ResultSet basis;

	private HashMap<String, LinkedHashMap<String, Double>> headPredictionsProviderValid;
	private HashMap<String, LinkedHashMap<String, Double>> tailPredictionsProviderValid;
	
	private HashMap<String, LinkedHashMap<String, Double>> headPredictionsProviderTest;
	private HashMap<String, LinkedHashMap<String, Double>> tailPredictionsProviderTest;
	
	private ConcurrentHashMap<String, AlphaBeta> relation2AlphaBeta = new ConcurrentHashMap<String, AlphaBeta>();

	
	public static void main(String[] args) throws IOException {
		
		if (args.length == 7) {
			
		}
		else {
			System.out.println("error: please specify 7 arguments in exactly this order seperated blanks");
			System.out.println(" 1) path to training file");
			System.out.println(" 2) path to validation file");
			System.out.println(" 3) path to ranking file created by a rule learner for the validation set");
			System.out.println(" 4) path to rule-based ranking file with scores from kge model for the validation set");
			System.out.println(" 5) path to ranking file created by a rule learner for the test set");
			System.out.println(" 6) path to rule-based ranking file with scores from kge model for the test set");
			System.out.println(" 7) output path whre the aggregated ranking file is stored");
			System.exit(1);
		}
		
		String training = args[0];
		String valid    = args[1];
		
		String rankingRulesValid = args[2];
		String rankingRulesTest  = args[3];
		
		String rankingKGEValid   = args[4];
		String rankingKGETest    = args[5];
		
		String rankingOutput     = args[6];
		
		TripleSet trainTS = new TripleSet(training);
		TripleSet validTS = new TripleSet(valid);
		
		Rescorer rs = new Rescorer(trainTS, validTS);
		
		ResultSet rulesValid = new ResultSet("rules valid", rankingRulesValid, true, 100);
		ResultSet kgeValid   = new ResultSet("KGE valid",   rankingKGEValid,   true, 100);
		
		ResultSet rulesTest = new ResultSet("rules test", rankingRulesTest, true, 100);
		ResultSet kgeTest   = new ResultSet("KGE test",   rankingKGETest,   true, 100);
		
		rs.searchJoinparameter(rulesValid, kgeValid, 4);
		
		rs.writeReorderedTestResult(rulesTest, kgeTest, rankingOutput);
		
		

	}
	
	public static void runBatch() throws IOException {
		
	
		
		// String[] models = new String[] {"complex", "conve", "hitter", "rescal", "transe", "tucker"}; // CODEX
		String[] models = new String[] {"complex", "conve", "distmult", "hitter", "rescal", "transe"}; // WN18RR & FB237
		
		// String[] models = new String[] {"hitter"};
		
		for (String kgeModel : models) {
			System.out.println("*******************************");
			System.out.println("*** going for " + kgeModel.toUpperCase() + " ***");
			System.out.println("*******************************");
			
			TripleSet train = new TripleSet("data/WN18RR/train.txt");
			TripleSet valid = new TripleSet("data/WN18RR/valid.txt");
			
			int numOfThreads = 4;
			
			Rescorer rs = new Rescorer(train, valid);
			
			String frag = "wn18rr/anyburl-c1-3600-100";
			
			// ResultSet kgeOnlyTest = new ResultSet("anyburl", "exp/understanding/codex-m/" + kgeModel + "-100-test", true, 100);
			
			ResultSet basisValid = new ResultSet("anyburl", "exp/understanding/" + frag + "-valid", true, 100);
			ResultSet providerValid = new ResultSet(kgeModel, "exp/understanding/" + frag + "-valid-" + kgeModel, true, 100);
			
			ResultSet basisTest = new ResultSet("anyburl", "exp/understanding/" + frag + "-test", true, 100);
			ResultSet providerTest = new ResultSet("kgeModel", "exp/understanding/" + frag + "-test-" + kgeModel, true, 100);
			
			rs.searchJoinparameter(basisValid, providerValid, numOfThreads);
			
			rs.writeReorderedTestResult(basisTest, providerTest, "exp/understanding/" + frag + "-test-" + kgeModel + "-MM");
		}
		
	}
	
	/**
	 * A re-scorer recomputes the scores or confidences in a candidate ranking by joining two result sets.
	 * 
	 * @param train Training set.
	 * @param valid Validation set used for fining the best join hyper parameter.
	 * @param test Test set to measure the quality of the resulting ranking.
	 */
	public Rescorer(TripleSet train, TripleSet valid) {
		this.train = train;
		this.valid = valid;
		
		this.headPredictionsProviderValid = new HashMap<String, LinkedHashMap<String, Double>>();
		this.tailPredictionsProviderValid = new HashMap<String, LinkedHashMap<String, Double>>();
		
		this.headPredictionsProviderTest = new HashMap<String, LinkedHashMap<String, Double>>();
		this.tailPredictionsProviderTest = new HashMap<String, LinkedHashMap<String, Double>>();
	}
	
	
	/**
	 * 
	 * @param basis A result set which has been computed against the validation set. Usually created by a rule based engine.
	 * @param scoreProviderValid Another results set which has been computed against the validation set. Usually created by a KGE approach.
	 * @throws IOException 
	 */
	public void searchJoinparameter(ResultSet basis, ResultSet scoreProviderValid, int numOfThreads) throws IOException {
		
		this.basis = basis;
		reorderProvider(scoreProviderValid, basis, true);
		System.out.println(">>> search for relation specific parameters");
		this.relations.addAll(this.train.getRelations());
		// HashMap<String, AlphaBeta> relation2AlphaBeta = new HashMap<String, AlphaBeta>();
		
		int r = 0;
		
		Thread[] rslearners = new Thread[numOfThreads];
		System.out.print(">>> creating worker threads ");
		for (int threadCounter = 0; threadCounter < 4; threadCounter++) {
			System.out.print("#" + threadCounter + " ");
			rslearners[threadCounter] = new RescoreLearner(this);
			rslearners[threadCounter].start();
		}
		System.out.println();
		
		while (alive(rslearners)) {
			try { Thread.sleep(500); }
			catch (InterruptedException e) { e.printStackTrace(); }
		}
		System.out.println(">>> all worker threads are done with their jobs");
		
		// for (String relation : relations) {
		// 	r = searchParameterForRelation(basis, relations, relation2AlphaBeta, r, relation);
		// }
		
		
	}
	
	synchronized public String getNextRelation()  {
		if (relations.size() > 0) {
			String relation = relations.poll();
			// System.out.println(">>> (" + relations.size() + " remaining) " + relation + " ");
			return relation;
		}
		return null;
	}

	public void searchParameterForRelation(String relation) throws IOException {
		
		TripleSet trainingSetR = new TripleSet();
		trainingSetR.addTriples(train.getTriplesByRelation(relation));
		
		ResultSet basisR = new ResultSet(basis, relation);
		

		TripleSet validationSetR = new TripleSet();
		validationSetR.addTriples(valid.getTriplesByRelation(relation));
		
		HitsAtK hitsAtK = new HitsAtK();
		
		hitsAtK.addFilterTripleSet(trainingSetR);
		hitsAtK.addFilterTripleSet(validationSetR);
		
		// double[] beta_values = new double[] {1.0}; 
		
		
		double[] beta_values = new double[] {0.0, 0.05, 0.1, 0.15, 0.2, 0.25, 0.3, 0.35, 0.4, 0.45, 0.5, 0.55, 0.6, 0.65, 0.7, 0.75, 0.8, 0.85, 0.9, 0.95, 1.0}; 
		
		// OLD double[] beta_values = new double[] {0.0, 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0}; 
		// double[] beta_values = new double[] {0.0, 0.5, 1.0}; 

		double[] best_beta = new double[2];
		double[] best_mrr = new double[2];
		double[] mrr = new double[2];
		for (double beta : beta_values) {
			// if (beta == 0.0 && alpha != 0) continue;
			// TODO here is the problem
			ResultSet rx = this.getRescored(basisR, beta);
			computeScores(rx, validationSetR, hitsAtK);
			mrr = new double[] {hitsAtK.getMRRHeads(), hitsAtK.getMRRTails()};
			for (int i = 0; i < 2; i++) {
				if (mrr[i] > best_mrr[i]) {
					// best_alpha[i] = alpha;
					best_beta[i] = beta;
					best_mrr[i] = mrr[i];
				}
			}
			hitsAtK.reset();
		}

		
		this.relation2AlphaBeta.put(relation + "+", new AlphaBeta(best_beta[0]));
		this.relation2AlphaBeta.put(relation + "-", new AlphaBeta(best_beta[1]));
		
		// if ((best_beta[0] <= 0.05 || best_beta[1] <= 0.05) && this.valid.getTriplesByRelation(relation).size() > 100) {
			System.out.println(">>> " + relation + "(" + this.valid.getTriplesByRelation(relation).size() + ") [+] " + best_mrr[0] + " based on beta=" + best_beta[0] + " [-] " + best_mrr[1] + " based on beta=" + best_beta[1]);
	
		// }
	}
	
	private static boolean alive(Thread[] threads) {
		for (Thread t : threads) {
			if (t.isAlive()) return true;
		}
		return false;
	}
	

	
	private void reorderProvider(ResultSet scoreProvider, ResultSet basis, boolean validNotTest) {
		System.out.println(">>> index and order the provider result set (" + (validNotTest ? "valid" : "test") +")");
		for (String triple : scoreProvider.getTriples()) {
			CompletionResult cr = scoreProvider.getCompletionResult(triple);
			LinkedHashMap<String, Double> headPredictions = new LinkedHashMap<String, Double>();
			LinkedHashMap<String, Double> tailPredictions = new LinkedHashMap<String, Double>();
			for (int i = 0; i < cr.getHeads().size(); i++) {
				String candidate = cr.getHeads().get(i);
				double score = cr.getHeadConfidences().get(i);
				headPredictions.put(candidate, score);
			}
			for (int i = 0; i < cr.getTails().size(); i++) {
				String candidate = cr.getTails().get(i);
				double score = cr.getTailConfidences().get(i);
				tailPredictions.put(candidate, score);
			}
			orderByValueDescending(headPredictions);
			orderByValueDescending(tailPredictions);
			
			// normalize01(headPredictions);
			// normalize01(tailPredictions);
			
			

			
			
			normalizeMinMax(headPredictions, basis.getHeadConfidences(triple));
			normalizeMinMax(tailPredictions, basis.getTailConfidences(triple));
			
			
			// TODO
			//int i = 1;
			// for (String c : headPredictions.keySet()) headPredictions.put(c, (double)i++);
			//for (String c : headPredictions.keySet()) headPredictions.put(c, 1.0 / (headPredictions.get(c) + 10.0));
			//i = 1;
			// for (String c : tailPredictions.keySet()) tailPredictions.put(c, (double)i++);
			//for (String c : tailPredictions.keySet()) tailPredictions.put(c, 1.0 / (tailPredictions.get(c) + 10.0));
			
			if (validNotTest) {
				this.headPredictionsProviderValid.put(triple,headPredictions);
				this.tailPredictionsProviderValid.put(triple,tailPredictions);
			}
			else {
				this.headPredictionsProviderTest.put(triple,headPredictions);
				this.tailPredictionsProviderTest.put(triple,tailPredictions);
			}
		}
	}
	
	/**
	 * Normalises the pre-ordered (highest first, lowest last) candidate ranking .
	 * Position based confidence approximation only.
	 * 
	 * #1 => 1 / 1 = 1.0
	 * #2 => 1 / 2 = 0.5 
	 * #3 => 1 / 3 = 0.333
	 * ... 
	 * 
	 * @param predictions
	 */
	private void normalizeP(LinkedHashMap<String, Double> predictions) {
		int i = 1;
		for (String c : predictions.keySet()) {
			predictions.put(c, 1.0 / (double)i);
			i++;
		}
		
	}
	
	/**
	 * Normalises the pre-ordered candidate ranking (highest first, lowest last).
	 * 
	 * Maps to the range [0, 1]. Highest will be 1, lowest will be 0.
	 * 
	 * @param predictions
	 */
	private void normalize01(LinkedHashMap<String, Double> predictions) {
		double max = -1000000;
		double min = +1000000;
		for (String c : predictions.keySet()) {
			double s = predictions.get(c);
			if (max < s) max = s;
			if (min > s) min = s;
		}
		for (String c : predictions.keySet()) {
			double s = predictions.get(c);
			double normalized = (s - min) / (max - min);
			if (normalized > 1.0) normalized = 1.0;
			if (normalized < 0.0) normalized = 0.0;
			predictions.put(c, normalized);
		}
	}
	
	/**
	 * Normalises the pre-ordered candidate ranking (highest first, lowest last).
	 * 
	 * Maps to the range [min, max]. Highest will be max, lowest will be min, where max is the highest confidence
	 * of the AnyBURL ranking and min is the minimal confidence of the AnyBURL ranking or 0, if the ranking if
	 * the AnyBURL ranming is incomplete.
	 * 
	 * @param predictions
	 */
	private void normalizeMinMax(LinkedHashMap<String, Double> predictions, ArrayList<Double> confidences) {
		
		
		double targetMax = confidences.size() > 0 ? confidences.get(0) : 0.01; // the second case should never occur
		double targetMin = confidences.size() == TOP_K ? confidences.get(TOP_K - 1) : 0.0;
		double targetSpan = targetMax - targetMin > 0 ? targetMax - targetMin : 0.01;
		
		double max = -1000000;
		double min = +1000000;
		for (String c : predictions.keySet()) {
			double s = predictions.get(c);
			if (max < s) max = s;
			if (min > s) min = s;
		}
		
		for (String c : predictions.keySet()) {
			double s = predictions.get(c);
			double normalized = (s - min) / (max - min);
			if (normalized > 1.0) normalized = 1.0;
			if (normalized < 0.0) normalized = 0.0;
			normalized = (normalized * targetSpan) + targetMin;
			predictions.put(c, normalized);
		}
	}
	
	
	
	private void writeReorderedTestResult(ResultSet basis, ResultSet scoreProvider, String outputPath) throws FileNotFoundException {
		this.reorderProvider(scoreProvider, basis, false);
		
		ResultSet rx = this.getRescored(basis);
		
		rx.write(outputPath);
		
		
	}
	
	// NOT USED ANYMORE
	private void fillUp(ResultSet rs, ResultSet filler) {
		for (CompletionResult cr : rs) {
			if (cr.getHeads().size() < TOP_K) {
				System.out.println("Fill up head: " + cr.getTripleAsString() + ": " + cr.getHeads().size());
				HashSet<String> heads = new HashSet<String>();
				for (String candidate : cr.getHeads()) heads.add(candidate);
				CompletionResult fr = filler.getCompletionResult(cr.getTripleAsString());
				for (int i = 0; i < TOP_K; i++)	{
					String fcandidate = fr.getHeads().get(i);
					if (!heads.contains(fcandidate)) {
						cr.getHeads().add(fcandidate);
						cr.getHeadConfidences().add(0.0);
					}
				}	
			}
			if (cr.getTails().size() < TOP_K) {
				System.out.println("Fill up tail: " + cr.getTripleAsString() + ": " + cr.getTails().size());
				HashSet<String> tails = new HashSet<String>();
				for (String candidate : cr.getTails()) tails.add(candidate);
				CompletionResult fr = filler.getCompletionResult(cr.getTripleAsString());
				for (int i = 0; i < TOP_K; i++)	{
					String fcandidate = fr.getTails().get(i);
					if (!tails.contains(fcandidate)) {
						cr.getTails().add(fcandidate);
						cr.getTailConfidences().add(0.0);
					}
				}	
			}
			
		}
		
		
	}
	
	
	private ResultSet getRescored(ResultSet basis, double beta) {

		ResultSet rs = new ResultSet();
		for (String tripleAsString : basis.getTriples()) {
			CompletionResult thisCr = basis.getCompletionResult(tripleAsString);
			CompletionResult reorderedCr = new CompletionResult(tripleAsString);

			ArrayList<String> reorderedHeads = new ArrayList<String>();
			ArrayList<Double> reorderedHeadConfidences = new ArrayList<Double>();
			reorderListsWeighted(reorderedHeads, reorderedHeadConfidences, thisCr.getHeads(), thisCr.getHeadConfidences(), this.headPredictionsProviderValid.get(tripleAsString), beta);
			reorderedCr.setHeads(reorderedHeads);
			reorderedCr.setHeadConfidences(reorderedHeadConfidences);
			
			ArrayList<String> reorderedTails = new ArrayList<String>();
			ArrayList<Double> reorderedTailConfidences = new ArrayList<Double>();
			reorderListsWeighted(reorderedTails, reorderedTailConfidences, thisCr.getTails(), thisCr.getTailConfidences(), this.tailPredictionsProviderValid.get(tripleAsString), beta);
			reorderedCr.setTails(reorderedTails);
			reorderedCr.setTailConfidences(reorderedTailConfidences);
			
			// this.results.put(tripleAsString, reorderedCr);
			
			rs.results.put(tripleAsString, reorderedCr);
			
		}
		return rs;
	}
	
	private ResultSet getRescored(ResultSet basis) {

		ResultSet rs = new ResultSet();
		for (String tripleAsString : basis.getTriples()) {
			String r = tripleAsString.split(" ")[1];
			AlphaBeta abH = relation2AlphaBeta.get(r + "+");
			AlphaBeta abT = relation2AlphaBeta.get(r + "-");
			
			CompletionResult thisCr = basis.getCompletionResult(tripleAsString);
			CompletionResult reorderedCr = new CompletionResult(tripleAsString);

			
			ArrayList<String> reorderedHeads = new ArrayList<String>();
			ArrayList<Double> reorderedHeadConfidences = new ArrayList<Double>();
			reorderListsWeighted(reorderedHeads, reorderedHeadConfidences, thisCr.getHeads(), thisCr.getHeadConfidences(), this.headPredictionsProviderTest.get(tripleAsString), abH.beta);
			reorderedCr.setHeads(reorderedHeads);
			reorderedCr.setHeadConfidences(reorderedHeadConfidences);
			
			ArrayList<String> reorderedTails = new ArrayList<String>();
			ArrayList<Double> reorderedTailConfidences = new ArrayList<Double>();
			reorderListsWeighted(reorderedTails, reorderedTailConfidences, thisCr.getTails(), thisCr.getTailConfidences(), this.tailPredictionsProviderTest.get(tripleAsString), abT.beta);
			reorderedCr.setTails(reorderedTails);
			reorderedCr.setTailConfidences(reorderedTailConfidences);
			
			// this.results.put(tripleAsString, reorderedCr);
			
			rs.results.put(tripleAsString, reorderedCr);
			
		}
		return rs;
	}
	
	private void reorderListsWeighted(ArrayList<String> reorderedCandidates, ArrayList<Double> reorderedConfidences, ArrayList<String> thisCandidates,  ArrayList<Double> thisConfidences, HashMap<String, Double> normalizedProvided, double beta) {
		LinkedHashMap<String, Double> map = new LinkedHashMap<String, Double>();
	
		
		
		for (int i = 0; i < thisCandidates.size(); i++) {
			String candidate =  thisCandidates.get(i);
			double confidence =  thisConfidences.get(i);		
			double normalizedScoreProvided  = normalizedProvided.get(candidate);
			
			double score = beta * confidence  + ((1.0 - beta) * normalizedScoreProvided); 
			// OLD: double score = beta * (confidence * ((1.0 + alpha) / (posInProvided + alpha))) + (1.0 - beta) * (1.0 / posInProvided); 
			// double score = beta * (confidence) + (1.0 - beta) * (1.0 / (1.0 + posInProvided)); 
			// System.out.println(candidate + ": " +  confidence + " => " + normalizedScoreProvided);
			
			map.put(candidate, score);
		}
		//System.out.println();
		orderByValueDescending(map);
		for (Map.Entry<String, Double> e : map.entrySet()) {
			reorderedCandidates.add(e.getKey());
			reorderedConfidences.add(e.getValue());
		}
		// System.out.println(reorderedCandidates.size());
	}


	private static void computeScores(ResultSet rs, TripleSet test, HitsAtK hitsAtK) {
		for (Triple t : test.getTriples()) {
				ArrayList<String> cand1 = rs.getHeadCandidates(t.toString());
				hitsAtK.evaluateHead(cand1, t);
				ArrayList<String> cand2 = rs.getTailCandidates(t.toString());
				hitsAtK.evaluateTail(cand2, t);
		}
	}
	
	private static void orderByValueDescending(LinkedHashMap<String, Double> m) {
	    List<Map.Entry<String, Double>> entries = new ArrayList<>(m.entrySet());
	    

	    Collections.sort(entries, new Comparator<Map.Entry<String, Double>>() {
	        @Override
	        public int compare(Map.Entry<String, Double> lhs, Map.Entry<String, Double> rhs) {
	        	if (lhs.getValue() - rhs.getValue() > 0) return -1;
	        	else {
	        		if (lhs.getValue() - rhs.getValue() == 0) return 0;
	        		else return 1;
	        	}
	        }
	    });

	    m.clear();
	    for(Map.Entry<String, Double> e : entries) {
	        m.put(e.getKey(), e.getValue());
	    }
	}



}
