# coding=utf-8
# Copyright 2024 The Qwen team, Alibaba Group and The HuggingFace Inc. team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Tokenization classes for Qwen2."""

import json
import os
import unicodedata
from functools import lru_cache
from typing import Optional, Tuple, List, Union

import regex as re

from ...tokenization_utils import AddedToken, PreTrainedTokenizer
from ...utils import logging


logger = logging.get_logger(__name__)

VOCAB_FILES_NAMES = {
    "vocab_file": "vocab.json",
    "merges_file": "merges.txt",
}


MAX_MODEL_INPUT_SIZES = {"qwen/qwen-tokenizer": 32768}

PRETOKENIZE_REGEX = r"""(?i:'s|'t|'re|'ve|'m|'ll|'d)|[^\r\n\p{L}\p{N}]?\p{L}+|\p{N}| ?[^\s\p{L}\p{N}]+[\r\n]*|\s*[\r\n]+|\s+(?!\S)|\s+"""


@lru_cache()
# Copied from transformers.models.gpt2.tokenization_gpt2.bytes_to_unicode
def bytes_to_unicode():
    """
    Returns list of utf-8 byte and a mapping to unicode strings. We specifically avoids mapping to whitespace/control
    characters the bpe code barfs on.

    The reversible bpe codes work on unicode strings. This means you need a large # of unicode characters in your vocab
    if you want to avoid UNKs. When you're at something like a 10B token dataset you end up needing around 5K for
    decent coverage. This is a significant percentage of your normal, say, 32K bpe vocab. To avoid that, we want lookup
    tables between utf-8 bytes and unicode strings.
    """
    bs = (
        list(range(ord("!"), ord("~") + 1)) + list(range(ord("¡"), ord("¬") + 1)) + list(range(ord("®"), ord("ÿ") + 1))
    )
    cs = bs[:]
    n = 0
    for b in range(2**8):
        if b not in bs:
            bs.append(b)
            cs.append(2**8 + n)
            n += 1
    cs = [chr(n) for n in cs]
    return dict(zip(bs, cs))


# Copied from transformers.models.gpt2.tokenization_gpt2.get_pairs
def get_pairs(word):
    """
    Return set of symbol pairs in a word.

    Word is represented as tuple of symbols (symbols being variable-length strings).
    """
    pairs = set()
    prev_char = word[0]
    for char in word[1:]:
        pairs.add((prev_char, char))
        prev_char = char
    return pairs


class Qwen2TokenizerBase(PreTrainedTokenizer):
    """
    Construct a Qwen2 tokenizer. Based on byte-level Byte-Pair-Encoding.

    Same with GPT2Tokenizer, this tokenizer has been trained to treat spaces like parts of the tokens so a word will
    be encoded differently whether it is at the beginning of the sentence (without space) or not:

    ```python
    >>> from transformers import Qwen2Tokenizer

    >>> tokenizer = Qwen2Tokenizer.from_pretrained("Qwen/Qwen-tokenizer")
    >>> tokenizer("Hello world")["input_ids"]
    [9707, 1879]

    >>> tokenizer(" Hello world")["input_ids"]
    [21927, 1879]
    ```
    This is expected.

    You should not use GPT2Tokenizer instead, because of the different pretokenization rules.

    This tokenizer inherits from [`PreTrainedTokenizer`] which contains most of the main methods. Users should refer to
    this superclass for more information regarding those methods.

    Args:
        vocab_file (`str`):
            Path to the vocabulary file.
        merges_file (`str`):
            Path to the merges file.
        errors (`str`, *optional*, defaults to `"replace"`):
            Paradigm to follow when decoding bytes to UTF-8. See
            [bytes.decode](https://docs.python.org/3/library/stdtypes.html#bytes.decode) for more information.
        unk_token (`str`, *optional*, defaults to `"<|endoftext|>"`):
            The unknown token. A token that is not in the vocabulary cannot be converted to an ID and is set to be this
            token instead.
        bos_token (`str`, *optional*):
            The beginning of sequence token. Not applicable for this tokenizer.
        eos_token (`str`, *optional*, defaults to `"<|endoftext|>"`):
            The end of sequence token.
        pad_token (`str`, *optional*, defaults to `"<|endoftext|>"`):
            The token used for padding, for example when batching sequences of different lengths.
        clean_up_tokenization_spaces (`bool`, *optional*, defaults to `False`):
            Whether or not the model should cleanup the spaces that were added when splitting the input text during the
            tokenization process. Not applicable to this tokenizer, since tokenization does not add spaces.
        split_special_tokens (`bool`, *optional*, defaults to `False`):
            Whether or not the special tokens should be split during the tokenization process. The default behavior is
            to not split special tokens. This means that if `<|endoftext|>` is the `eos_token`, then `tokenizer.tokenize("<|endoftext|>") =
            ['<|endoftext|>`]. Otherwise, if `split_special_tokens=True`, then `tokenizer.tokenize("<|endoftext|>")` will be give `['<',
            '|', 'endo', 'ft', 'ext', '|', '>']`. This argument is only supported for `slow` tokenizers for the moment.
    """

    vocab_files_names = VOCAB_FILES_NAMES
    model_input_names = ["input_ids", "attention_mask"]

    def __init__(
        self,
        vocab_file,
        merges_file,
        errors="replace",
        unk_token="<|endoftext|>",
        bos_token=None,
        eos_token="<|endoftext|>",
        pad_token="<|endoftext|>",
        clean_up_tokenization_spaces=False,
        split_special_tokens=False,
        **kwargs,
    ):
        # Qwen vocab does not contain control tokens; added tokens need to be special
        bos_token = (
            AddedToken(bos_token, lstrip=False, rstrip=False, special=True, normalized=False)
            if isinstance(bos_token, str)
            else bos_token
        )
        eos_token = (
            AddedToken(eos_token, lstrip=False, rstrip=False, special=True, normalized=False)
            if isinstance(eos_token, str)
            else eos_token
        )
        unk_token = (
            AddedToken(unk_token, lstrip=False, rstrip=False, special=True, normalized=False)
            if isinstance(unk_token, str)
            else unk_token
        )
        pad_token = (
            AddedToken(pad_token, lstrip=False, rstrip=False, special=True, normalized=False)
            if isinstance(pad_token, str)
            else pad_token
        )

        with open(vocab_file, encoding="utf-8") as vocab_handle:
            self.encoder = json.load(vocab_handle)
        self.decoder = {v: k for k, v in self.encoder.items()}
        self.errors = errors  # how to handle errors in decoding
        self.byte_encoder = bytes_to_unicode()
        self.byte_decoder = {v: k for k, v in self.byte_encoder.items()}
        bpe_merges = []
        with open(merges_file, encoding="utf-8") as merges_handle:
            for i, line in enumerate(merges_handle):
                line = line.strip()
                if (i == 0 and line.startswith("#version:")) or not line:
                    continue
                bpe_merges.append(tuple(line.split()))
        self.bpe_ranks = dict(zip(bpe_merges, range(len(bpe_merges))))
        # NOTE: the cache can grow without bound and will get really large for long running processes
        # (esp. for texts of language that do not use space between word, e.g. Chinese); technically
        # not a memory leak but appears as one.
        # GPT2Tokenizer has the same problem, so let's be consistent.
        self.cache = {}

        self.pat = re.compile(PRETOKENIZE_REGEX)

        if kwargs.get("add_prefix_space", False):
            logger.warning_once(
                f"{self.__class__.__name} does not support `add_prefix_space`, setting it to True has no effect."
            )

        super().__init__(
            errors=errors,
            bos_token=bos_token,
            eos_token=eos_token,
            pad_token=pad_token,
            unk_token=unk_token,
            clean_up_tokenization_spaces=clean_up_tokenization_spaces,
            split_special_tokens=split_special_tokens,
            **kwargs,
        )

    @property
    def vocab_size(self) -> int:
        return len(self.encoder)

    # Copied from transformers.models.gpt2.tokenization_gpt2.GPT2Tokenizer.get_vocab
    def get_vocab(self):
        return dict(self.encoder, **self.added_tokens_encoder)

    # Copied from transformers.models.gpt2.tokenization_gpt2.GPT2Tokenizer.bpe
    def bpe(self, token):
        if token in self.cache:
            return self.cache[token]
        word = tuple(token)
        pairs = get_pairs(word)

        if not pairs:
            return token

        while True:
            bigram = min(pairs, key=lambda pair: self.bpe_ranks.get(pair, float("inf")))
            if bigram not in self.bpe_ranks:
                break
            first, second = bigram
            new_word = []
            i = 0
            while i < len(word):
                try:
                    j = word.index(first, i)
                except ValueError:
                    new_word.extend(word[i:])
                    break
                else:
                    new_word.extend(word[i:j])
                    i = j

                if word[i] == first and i < len(word) - 1 and word[i + 1] == second:
                    new_word.append(first + second)
                    i += 2
                else:
                    new_word.append(word[i])
                    i += 1
            new_word = tuple(new_word)
            word = new_word
            if len(word) == 1:
                break
            else:
                pairs = get_pairs(word)
        word = " ".join(word)
        self.cache[token] = word
        return word

    # Copied from transformers.models.gpt2.tokenization_gpt2.GPT2Tokenizer._tokenize
    def _tokenize(self, text):
        """Tokenize a string."""
        bpe_tokens = []
        for token in re.findall(self.pat, text):
            token = "".join(
                self.byte_encoder[b] for b in token.encode("utf-8")
            )  # Maps all our bytes to unicode strings, avoiding control tokens of the BPE (spaces in our case)
            bpe_tokens.extend(bpe_token for bpe_token in self.bpe(token).split(" "))
        return bpe_tokens

    # Copied from transformers.models.gpt2.tokenization_gpt2.GPT2Tokenizer._convert_token_to_id
    def _convert_token_to_id(self, token):
        """Converts a token (str) in an id using the vocab."""
        return self.encoder.get(token, self.encoder.get(self.unk_token))

    # Copied from transformers.models.gpt2.tokenization_gpt2.GPT2Tokenizer._convert_id_to_token
    def _convert_id_to_token(self, index):
        """Converts an index (integer) in a token (str) using the vocab."""
        return self.decoder.get(index)

    # Copied from transformers.models.gpt2.tokenization_gpt2.GPT2Tokenizer.convert_tokens_to_string
    def convert_tokens_to_string(self, tokens):
        """Converts a sequence of tokens (string) in a single string."""
        text = "".join(tokens)
        text = bytearray([self.byte_decoder[c] for c in text]).decode("utf-8", errors=self.errors)
        return text

    def decode(
        self,
        token_ids,
        skip_special_tokens: bool = False,
        clean_up_tokenization_spaces: Optional[bool] = False,
        spaces_between_special_tokens: bool = False,
        **kwargs,
    ) -> str:
        # `spaces_between_special_tokens` defaults to True for _decode in slow tokenizers
        # and cannot be configured elsewhere, but it should default to False for Qwen2Tokenizer
        return super().decode(
            token_ids,
            skip_special_tokens=skip_special_tokens,
            clean_up_tokenization_spaces=clean_up_tokenization_spaces,
            spaces_between_special_tokens=spaces_between_special_tokens,
            **kwargs,
        )

    # Copied from transformers.models.gpt2.tokenization_gpt2.GPT2Tokenizer.save_vocabulary
    def save_vocabulary(self, save_directory: str, filename_prefix: Optional[str] = None) -> Tuple[str]:
        if not os.path.isdir(save_directory):
            logger.error(f"Vocabulary path ({save_directory}) should be a directory")
            return
        vocab_file = os.path.join(
            save_directory, (filename_prefix + "-" if filename_prefix else "") + VOCAB_FILES_NAMES["vocab_file"]
        )
        merge_file = os.path.join(
            save_directory, (filename_prefix + "-" if filename_prefix else "") + VOCAB_FILES_NAMES["merges_file"]
        )

        with open(vocab_file, "w", encoding="utf-8") as f:
            f.write(json.dumps(self.encoder, indent=2, sort_keys=True, ensure_ascii=False) + "\n")

        index = 0
        with open(merge_file, "w", encoding="utf-8") as writer:
            writer.write("#version: 0.2\n")
            for bpe_tokens, token_index in sorted(self.bpe_ranks.items(), key=lambda kv: kv[1]):
                if index != token_index:
                    logger.warning(
                        f"Saving vocabulary to {merge_file}: BPE merge indices are not consecutive."
                        " Please check that the tokenizer is not corrupted!"
                    )
                    index = token_index
                writer.write(" ".join(bpe_tokens) + "\n")
                index += 1

        return vocab_file, merge_file

    def prepare_for_tokenization(self, text, **kwargs):
        text = unicodedata.normalize("NFC", text)
        return (text, kwargs)

"""
Qwen2 Tokenizer with parameter token support.

This module extends the base Qwen2 tokenizer to handle special parameter tokens
for values in the range [2048, 3072).
"""

import torch

class Qwen2Tokenizer(Qwen2TokenizerBase):
    """
    Qwen2 tokenizer with support for parameter tokens.
    
    This tokenizer extends the base Qwen2 tokenizer to handle special parameter
    tokens that map integer values in the range [2048, 3072) to dedicated token IDs.
    
    Args:
        vocab_file (str): Path to the vocabulary file.
        merges_file (str): Path to the merges file.
        param_token_start (int, optional): Starting token ID for parameter tokens. 
            Defaults to 100000.
        param_token_count (int, optional): Number of parameter tokens to reserve.
            Defaults to 1024.
        **kwargs: Additional keyword arguments passed to the parent class.
        
    Raises:
        AssertionError: If the vocabulary size is insufficient for parameter tokens.
    """
    
    PARAM_VALUE_MIN = 2048
    PARAM_VALUE_MAX = 3072
    DEFAULT_PARAM_TOKEN_START = 100000
    DEFAULT_PARAM_TOKEN_COUNT = 1024
    
    def __init__(
        self,
        vocab_file: str,
        merges_file: str,
        param_token_start: int = DEFAULT_PARAM_TOKEN_START,
        param_token_count: int = DEFAULT_PARAM_TOKEN_COUNT,
        **kwargs,
    ) -> None:
        super().__init__(vocab_file, merges_file, **kwargs)
        self.param_token_start = param_token_start
        self.param_token_count = param_token_count
        
        self._validate_vocabulary_size()
    
    def _validate_vocabulary_size(self) -> None:
        """Validate that vocabulary is large enough for parameter tokens."""
        required_size = self.param_token_start + self.param_token_count
        if required_size > len(self.encoder):
            raise ValueError(
                f"Vocabulary size {len(self.encoder)} is insufficient for parameter "
                f"tokens. Required: {required_size}"
            )
    
    def _convert_param_to_token_id(self, param_value: int) -> Optional[int]:
        """
        Convert a parameter value to its corresponding token ID.
        
        Args:
            param_value (int): Parameter value to convert.
            
        Returns:
            Optional[int]: Token ID if value is in valid range, None otherwise.
        """
        if self.PARAM_VALUE_MIN <= param_value < self.PARAM_VALUE_MAX:
            return self.param_token_start + (param_value - self.PARAM_VALUE_MIN)
        return None
    
    def _convert_token_id_to_param(self, token_id: int) -> Optional[int]:
        """
        Convert a token ID back to its parameter value.
        
        Args:
            token_id (int): Token ID to convert.
            
        Returns:
            Optional[int]: Parameter value if token ID is a parameter token, None otherwise.
        """
        if self._is_param_token_id(token_id):
            return self.PARAM_VALUE_MIN + (token_id - self.param_token_start)
        return None
    
    def _is_param_value(self, text: str) -> bool:
        """
        Check if text represents a valid parameter value.
        
        Args:
            text (str): Text to check.
            
        Returns:
            bool: True if text is a valid parameter value, False otherwise.
        """
        try:
            value = int(text)
            return self.PARAM_VALUE_MIN <= value < self.PARAM_VALUE_MAX
        except (ValueError, TypeError):
            return False
    
    def _is_param_token_id(self, token_id: int) -> bool:
        """
        Check if token ID is a parameter token.
        
        Args:
            token_id (int): Token ID to check.
            
        Returns:
            bool: True if token ID is a parameter token, False otherwise.
        """
        return (self.param_token_start <= token_id < 
                self.param_token_start + self.param_token_count)
    
    def _tokenize(self, text: str) -> List[str]:
        """
        Tokenize text with parameter token support.
        
        Args:
            text (str): Text to tokenize.
            
        Returns:
            List[str]: List of tokens.
        """
        tokens = []
        # Split on digit sequences to identify potential parameter values
        segments = re.split(r'(\d+)', text)
        for segment in segments:
            if not segment:
                continue
            if self._is_param_value(segment):
                param_value = int(segment)
                token_id = self._convert_param_to_token_id(param_value)
                if token_id is not None:
                    # Use the decoder to get the token representation
                    tokens.append(self.decoder[token_id])
                    continue
            # Use parent class tokenization for non-parameter segments
            tokens.extend(super()._tokenize(segment))
        return tokens
    
    def _convert_token_to_id(self, token: str) -> int:
        """Convert token to ID using parent implementation."""
        return super()._convert_token_to_id(token)
    
    def _convert_id_to_token(self, index: int) -> str:
        """Convert ID to token using parent implementation."""
        return super()._convert_id_to_token(index)
    
    def convert_tokens_to_string(self, tokens: List[str]) -> str:
        """
        Convert tokens to string, filtering out empty tokens.
        
        Args:
            tokens (List[str]): List of tokens to convert.
            
        Returns:
            str: Decoded string.
        """
        # Filter out empty tokens
        filtered_tokens = [token for token in tokens if token]
        return super().convert_tokens_to_string(filtered_tokens)
    
    def encode(self, text: str, **kwargs) -> List[int]:
        """
        Encode text to token IDs with parameter token support.
        
        Args:
            text (str): Text to encode.
            **kwargs: Additional keyword arguments.
            
        Returns:
            List[int]: List of token IDs.
        """
        # Handle single parameter values directly
        if self._is_param_value(text):
            param_value = int(text)
            token_id = self._convert_param_to_token_id(param_value)
            if token_id is not None:
                return [token_id]
        return super().encode(text, **kwargs)
    
    def decode(
        self,
        token_ids: Union[List[int], torch.Tensor],
        skip_special_tokens: bool = False,
        clean_up_tokenization_spaces: Optional[bool] = False,
        spaces_between_special_tokens: bool = False,
        **kwargs,
    ) -> str:
        """
        Decode token IDs to text with parameter token support.
        
        Args:
            token_ids (Union[List[int], torch.Tensor]): Token IDs to decode.
            skip_special_tokens (bool, optional): Whether to skip special tokens.
                Defaults to False.
            clean_up_tokenization_spaces (Optional[bool], optional): Whether to clean
                up tokenization spaces. Defaults to False.
            spaces_between_special_tokens (bool, optional): Whether to add spaces
                between special tokens. Defaults to False.
            **kwargs: Additional keyword arguments.
            
        Returns:
            str: Decoded text.
        """
        # Convert tensor to list if necessary
        if isinstance(token_ids, torch.Tensor):
            token_ids = token_ids.tolist()
        decoded_tokens = []
        for token_id in token_ids:
            param_value = self._convert_token_id_to_param(token_id)
            if param_value is not None:
                # Convert parameter token back to its numeric value
                decoded_tokens.append(str(param_value))
            else:
                # Use standard token conversion
                decoded_tokens.append(self._convert_id_to_token(token_id))
        return self.convert_tokens_to_string(decoded_tokens)
