import torch
import torch.nn as nn
from operations import *
from torch.autograd import Variable
from utils import drop_path


class ECALayer(nn.Module):
    def __init__(self, channel, k_size=3):
        super(ECALayer, self).__init__()

        self.avg_pool = nn.AdaptiveAvgPool2d(1)

        self.conv = nn.Conv1d(1, 1, kernel_size=k_size, padding=(k_size - 1) // 2, bias=False)
        self.sigmoid = nn.Sigmoid()

    def forward(self, x):
        b, c, _, _ = x.size()

        y = self.avg_pool(x).view(b, 1, c)

        y = self.conv(y)

        y = self.sigmoid(y).view(b, c, 1, 1)
        return x * y.expand_as(x)


class DualBranchCell(nn.Module):
    def __init__(self, genotype, multiplier, C_prev_prev, C_prev, C, reduction, reduction_prev):
        super(DualBranchCell, self).__init__()

        self.reduction = False
        self.multiplier = 4
        self.cell_robust = Cell(genotype, C_prev_prev, C_prev, C, reduction, reduction_prev, False)
        self.cell_normal = Cell(genotype, C_prev_prev, C_prev, C, reduction, reduction_prev, True)

        self.fusion = nn.Sequential(
            nn.ReLU(inplace=False),
            nn.Conv2d(2 * multiplier * C, multiplier * C, kernel_size=1, bias=False),
            nn.BatchNorm2d(multiplier * C),
            ECALayer(multiplier * C, k_size=3)
        )

    def forward(self, states, drop_path_prob, cell_index):
        states_normal = self.cell_normal(states, drop_path_prob, cell_index)
        states_robust = self.cell_robust(states, drop_path_prob, cell_index)

        robust_feature = states_robust[-1]['feature']
        normal_feature = states_normal[-1]['feature']

        fused_feature = torch.cat([robust_feature, normal_feature], dim=1)

        fused_feature = self.fusion(fused_feature)

        new_state = {'feature': fused_feature, 'cell_index': cell_index, 'node_index': -1}
        return states + [new_state]

    def wider(self, k):
        self.cell_robust.wider(k)
        self.cell_normal.wider(k)


class Cell(nn.Module):

    def __init__(self, genotype, C_prev_prev, C_prev, C, reduction, reduction_prev, normal):
        super(Cell, self).__init__()
        # print(C_prev_prev, C_prev, C)
        self.reduction = reduction

        if reduction_prev:
            self.preprocess0 = FactorizedReduce(C_prev_prev, C)
        else:
            self.preprocess0 = ReLUConvBN(C_prev_prev, C, 1, 1, 0)  # 256
        self.preprocess1 = ReLUConvBN(C_prev, C, 1, 1, 0)  # 512

        if reduction:
            op_names, indices = zip(*genotype.reduce)
            concat = genotype.reduce_concat
        elif normal:
            op_names, indices = zip(*genotype.normal)
            concat = genotype.normal_concat
        else:
            op_names, indices = zip(*genotype.robust)
            concat = genotype.robust_concat
        self._compile(C, op_names, indices, concat, reduction)

    def _compile(self, C, op_names, indices, concat, reduction):
        assert len(op_names) == len(indices)
        self._steps = len(op_names) // 2
        self._concat = concat
        self.multiplier = len(concat)

        self._ops = nn.ModuleList()
        for name, index in zip(op_names, indices):
            stride = 2 if reduction and index < 2 else 1
            op = OPS[name](C, stride, True)
            self._ops += [op]
        self._indices = indices

    def forward(self, states, drop_prob, cell_index):

        state = states[cell_index - 1]

        if cell_index == 1:
            s0 = self.preprocess0(state[-2]['feature'])
            s1 = self.preprocess1(state[-1]['feature'])
        else:
            s0 = self.preprocess0(states[cell_index - 2][-1]['feature'])
            s1 = self.preprocess1(states[cell_index - 1][-1]['feature'])

        all_states = [{'feature': s0, 'cell_index': cell_index, 'node_index': -1},
                      {'feature': s1, 'cell_index': cell_index, 'node_index': -1}]

        states = [s0, s1]
        k = 0
        for i in range(self._steps):
            h1 = all_states[self._indices[2 * i]]['feature']
            h2 = all_states[self._indices[2 * i + 1]]['feature']
            op1 = self._ops[2 * i]
            op2 = self._ops[2 * i + 1]
            h1 = op1(h1)
            h2 = op2(h2)
            if self.training and drop_prob > 0.:
                if not isinstance(op1, Identity):
                    h1 = drop_path(h1, drop_prob)
                if not isinstance(op2, Identity):
                    h2 = drop_path(h2, drop_prob)
            s = h1 + h2
            all_states.append({'feature': s, 'cell_index': cell_index, 'node_index': i})
            k += 1
        k += 1
        result = torch.cat([all_states[i]['feature'] for i in self._concat], dim=1)
        all_states.append({'feature': result, 'cell_index': cell_index, 'node_index': k})
        return all_states


class NetworkCIFAR(nn.Module):

    def __init__(self, C, num_classes, layers, auxiliary, genotype, drop_path_prob=0):
        super(NetworkCIFAR, self).__init__()
        self._layers = layers
        self._auxiliary = auxiliary
        self.drop_path_prob = drop_path_prob
        stem_multiplier = 4
        C_curr = stem_multiplier * C  # 256
        self.stem = nn.Sequential(
            nn.Conv2d(3, C_curr, 3, padding=1, bias=False),
            nn.BatchNorm2d(C_curr)
        )

        C_prev_prev, C_prev, C_curr = C_curr, C_curr, C  # 256 256 64  i=0-2
        self.cells = nn.ModuleList()
        reduction_prev = False
        for i in range(layers):
            if i in [layers // 3, 2 * layers // 3]:
                C_curr *= 2
                reduction = True

                cell = Cell(genotype, C_prev_prev, C_prev, C_curr, reduction, reduction_prev, True)

            else:
                reduction = False
                cell = DualBranchCell(genotype, stem_multiplier, C_prev_prev, C_prev, C_curr, reduction, reduction_prev)

            reduction_prev = reduction
            self.cells.append(cell)
            C_prev_prev, C_prev = C_prev, cell.multiplier * C_curr

        self.global_pooling = nn.AdaptiveAvgPool2d(1)
        self.classifier = nn.Linear(C_prev, num_classes)

    def forward(self, input, return_features=False, cp_loss=False):
        logits_aux = None
        s0 = s1 = self.stem(input)
        cell_front_1 = 1

        state = [{'feature': s0, 'cell_index': -1, 'node_index': 0},
                 {'feature': s1, 'cell_index': -1, 'node_index': 1}]
        cell_tick = 0
        states = [state]
        features = []

        for i, cell in enumerate(self.cells):
            updated_states = cell(states, self.drop_path_prob, cell_index=i + 1)
            cell_tick += 1

            states.append(updated_states)

        out = self.global_pooling(states[-1][-1]['feature'])
        logits = self.classifier(out.view(out.size(0), -1))

        return logits


class NetworkImageNet(nn.Module):

    def __init__(self, C, num_classes, layers, auxiliary, genotype):
        super(NetworkImageNet, self).__init__()
        self._layers = layers
        self._auxiliary = auxiliary

        self.stem0 = nn.Sequential(
            nn.Conv2d(3, C // 2, kernel_size=3, stride=2, padding=1, bias=False),
            nn.BatchNorm2d(C // 2),
            nn.ReLU(inplace=True),
            nn.Conv2d(C // 2, C, 3, stride=2, padding=1, bias=False),
            nn.BatchNorm2d(C),
        )

        self.stem1 = nn.Sequential(
            nn.ReLU(inplace=True),
            nn.Conv2d(C, C, 3, stride=2, padding=1, bias=False),
            nn.BatchNorm2d(C),
        )

        C_prev_prev, C_prev, C_curr = C, C, C

        self.cells = nn.ModuleList()
        reduction_prev = True
        for i in range(layers):
            if i in [layers // 3, 2 * layers // 3]:
                C_curr *= 2
                reduction = True
            else:
                reduction = False
            cell = Cell(genotype, C_prev_prev, C_prev, C_curr, reduction, reduction_prev)
            reduction_prev = reduction
            self.cells += [cell]
            C_prev_prev, C_prev = C_prev, cell.multiplier * C_curr
            if i == 2 * layers // 3:
                C_to_auxiliary = C_prev

        if auxiliary:
            self.auxiliary_head = AuxiliaryHeadImageNet(C_to_auxiliary, num_classes)
        self.global_pooling = nn.AvgPool2d(7)
        self.classifier = nn.Linear(C_prev, num_classes)

    def forward(self, input):
        logits_aux = None
        s0 = self.stem0(input)
        s1 = self.stem1(s0)
        for i, cell in enumerate(self.cells):
            s0, s1 = s1, cell(s0, s1, self.drop_path_prob)
            if i == 2 * self._layers // 3:
                if self._auxiliary and self.training:
                    logits_aux = self.auxiliary_head(s1)
        out = self.global_pooling(s1)
        logits = self.classifier(out.view(out.size(0), -1))
        return logits  # logits_aux

