#!/usr/bin/env python3

import click
import logging
import re
import subprocess
import sys

from src.answers import check_answer
from src.models import *
from src.suites import SUITES
from src.prompt import create_python_code_prompt


logging.basicConfig(
    stream=sys.stdout,
    level=logging.INFO,
    format="[%(asctime)s] %(levelname)s :: %(message)s",
)
logger = logging.getLogger(__name__)


def validate_temperature(ctx, param, value):
    if value > 2 or value < 0:
        raise click.BadParameter("The temperature must be in the interval [0,2].")
    return value

def validate_top_p(ctx, param, value):
    if value > 1 or value < 0:
        raise click.BadParameter("The top-P must be in the interval [0,1].")
    return value

def extract_code(answer) -> str:
    if "```" in answer:
        return re.search(r"```(?:\w+\n)?(.*?)```", answer, re.DOTALL).group(1)
    elif "<code>" in answer:
        return re.search(r"<code>(?:\w+\n)?(.*?)</code>", answer, re.DOTALL).group(1)
    else:
        return answer


@click.command()
@click.option(
    "--instance",
    "-i",
    required=True,
    type=str,
    help="Instance used.",
)
@click.option(
    "--code-file",
    required=True,
    type=str,
    help="File where code will be written.",
)
@click.option(
    "--framework",
    default="gemini",
    type=click.Choice(["gemini", "deepseek", "openai"]),
    help="Framework used to run LLMs."
)
@click.option(
    "--model",
    "-m",
    default="gemini-2.0-flash-thinking",
    help="LLM Model used.",
)
@click.option(
    "--temperature",
    "-t",
    default=0.6,
    type=float,
    callback=validate_temperature,
    help="Model temperature.",
)
@click.option(
    "--top-p",
    default=0.95,
    type=float,
    callback=validate_top_p,
    help="Model top-P.",
)
@click.option(
    "--only-prompt",
    is_flag=True,
    help="Exit program after generating prompt.")
@click.option(
    "--dimacs",
    is_flag=True,
    help="Code generation for dimacs.")
def main(instance, code_file, framework, model, temperature, top_p, only_prompt, dimacs):
    logging.info(f"Python version: {sys.version}.")
    logging.info(f"Using model {model}.")
    logging.info(f"Using prompt python code.")
    logging.info(f"Using temperature {temperature}.")
    logging.info(f"Using top-P {top_p}.")
    if dimacs:
        logging.info(f"Using DIMACS: {dimacs}.")

    prompt = create_python_code_prompt(instance, dimacs)
    logging.info("Final prompt: ")
    print(prompt)

    if only_prompt:
        logging.info("Exitting after prompt was generated.")
        return

    if framework == "gemini":
        thinking_content, answer = run_gemini(model, prompt, temperature, top_p)
    elif framework == "deepseek":
        answer = run_deepseek(model, prompt, temperature, top_p)
    elif framework == "openai":
        answer = run_openai(model, prompt, temperature, top_p)

    logging.info("LLM Answer:")
    print(answer)

    if len(answer) == 0:
        logging.error("LLM returned an empty string!")
        raise ValueError("LLM answer is empty.")

    code = extract_code(answer)
    logging.info("Code extracted:")
    print(code)

    logging.info(f"Code: {code}")

    with open(code_file, "w") as f:
        logging.info(f"Saving code to {code_file}")
        f.write(code)

    logging.info(f"Running code from {code_file}")
    try:
        with open(instance, "r") as f:
            instance_content = f.read()

        process = subprocess.run(
            ["python3", code_file],
            input=instance_content,
            capture_output=True,
            text=True,
            check=True,
            timeout=300,
        )
        output = process.stdout.strip()
        logging.info(f"Code output: {output}")

        valid_answer = check_answer(instance, output)
        logging.info(f"Valid answer: {valid_answer}")

    except subprocess.CalledProcessError as e:
        logging.error(f"Error running generated code: {e}")
        logging.error(f"Stderr: {e.stderr}")
    except subprocess.TimeoutExpired as e:
        logging.error(f"Code execution timed out: {e}")
    except FileNotFoundError:
        logging.error(f"Instance file not found at {instance}")


    logging.info("Finished correctly.")


if __name__ == "__main__":
    main()
