#!/usr/bin/env python3

import click
import logging
import subprocess
import sys

from src.answers import check_answer
from src.models import *
from src.suites import SUITES
from src.prompt import create_end_to_end_prompt


logging.basicConfig(
    stream=sys.stdout,
    level=logging.INFO,
    format="[%(asctime)s] %(levelname)s :: %(message)s",
)
logger = logging.getLogger(__name__)


def validate_temperature(ctx, param, value):
    if value > 2 or value < 0:
        raise click.BadParameter("The temperature must be in the interval [0,2].")
    return value

def validate_top_p(ctx, param, value):
    if value > 1 or value < 0:
        raise click.BadParameter("The top-P must be in the interval [0,1].")
    return value


@click.command()
@click.option(
    "--instance",
    "-i",
    required=True,
    type=str,
    help="Instance used.",
)
@click.option(
    "--framework",
    default="gemini",
    type=click.Choice(["gemini", "deepseek", "openai"]),
    help="Framework used to run LLMs."
)
@click.option(
    "--model",
    "-m",
    default="gemini-2.0-flash-thinking",
    help="LLM Model used.",
)
@click.option(
    "--temperature",
    "-t",
    default=0.6,
    type=float,
    callback=validate_temperature,
    help="Model temperature.",
)
@click.option(
    "--top-p",
    default=0.95,
    type=float,
    callback=validate_top_p,
    help="Model top-P.",
)
@click.option(
    "--prompt-format",
    default="base",
    type=click.Choice(["base", "kc", "sudoku_free"]),
    help="Format of the prompt.",
)
@click.option(
    "--only-prompt",
    is_flag=True,
    help="Exit program after generating prompt.")
def main(instance, framework, model, temperature, top_p, prompt_format, only_prompt):
    logging.info(f"Python version: {sys.version}.")
    logging.info(f"Using model {model}.")
    logging.info(f"Using prompt {prompt_format}.")
    logging.info(f"Using temperature {temperature}.")
    logging.info(f"Using top-P {top_p}.")

    prompt = create_end_to_end_prompt(instance, prompt_format)
    logging.info("Final prompt: ")
    print(prompt)

    if only_prompt:
        logging.info("Exitting after prompt was generated.")
        return

    if framework == "gemini":
        thinking_content, answer = run_gemini(model, prompt, temperature, top_p)
    elif framework == "deepseek":
        answer = run_deepseek(model, prompt, temperature, top_p)
    elif framework == "openai":
        answer = run_openai(model, prompt, temperature, top_p)

    logging.info("LLM Answer:")
    print(answer)

    if len(answer) == 0:
        logging.error("LLM returned an empty string!")
        raise ValueError("LLM answer is empty.")

    valid_answer = check_answer(instance, answer)
    logging.info(f"Valid answer: {valid_answer}")


    logging.info("Finished correctly.")


if __name__ == "__main__":
    main()
