import argparse
import random

def read_dimacs_cnf(filename):
    with open(filename, 'r') as f:
        lines = f.readlines()

    comments = []
    problem_line = None
    clauses = []

    for line in lines:
        line = line.strip()
        if line.startswith('c'):
            comments.append(line)
        elif line.startswith('p'):
            problem_line = line
        elif line:
            clause = [int(x) for x in line.split() if x != '0']
            if clause:
                clauses.append(clause)

    return comments, problem_line, clauses

def write_dimacs_cnf(filename, comments, num_vars, clauses):
    with open(filename, 'w') as f:
        for c in comments:
            f.write(f"{c}\n")
        f.write(f"p cnf {num_vars} {len(clauses)}\n")
        for clause in clauses:
            clause_line = ' '.join(str(lit) for lit in clause) + ' 0\n'
            f.write(clause_line)

def permute_variables(clauses, num_vars, seed=None):
    if seed is not None:
        random.seed(seed)
    
    permutation = list(range(1, num_vars + 1))
    random.shuffle(permutation)

    # Map original var -> permuted var
    var_map = {orig: new for orig, new in zip(range(1, num_vars + 1), permutation)}

    new_clauses = []
    for clause in clauses:
        new_clause = []
        for lit in clause:
            var = abs(lit)
            sign = 1 if lit > 0 else -1
            new_var = var_map[var]
            new_clause.append(sign * new_var)
        new_clauses.append(new_clause)

    return new_clauses

def permute_clauses(clauses, seed=None):
    if seed is not None:
        random.seed(seed)
    random.shuffle(clauses)
    return clauses

def get_num_vars(problem_line):
    return int(problem_line.split()[2])

def main():
    parser = argparse.ArgumentParser(description='Permute DIMACS CNF variables and optionally clauses.')
    parser.add_argument('input', help='Input DIMACS CNF file')
    parser.add_argument('-o', '--output', default='/dev/stdout', help='Output DIMACS CNF file')
    parser.add_argument('--permute-clauses', action='store_true', help='Permute clause order')
    parser.add_argument('--seed', type=int, help='Random seed for reproducibility')

    args = parser.parse_args()

    comments, problem_line, clauses = read_dimacs_cnf(args.input)
    num_vars = get_num_vars(problem_line)

    clauses = permute_variables(clauses, num_vars, seed=args.seed)

    if args.permute_clauses:
        clauses = permute_clauses(clauses, seed=args.seed)

    write_dimacs_cnf(args.output, comments, num_vars, clauses)
    print(f"c Permuted CNF written to {args.output}")

if __name__ == '__main__':
    main()
