# coding=utf-8
# Copyright 2023-present the HuggingFace Inc. team.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
import math
import enum
import re
import warnings
from dataclasses import asdict, dataclass, field, replace
from enum import Enum
from typing import List, Optional, Tuple, Union

import torch
import torch.nn as nn
import torch.nn.functional as F
from transformers.pytorch_utils import Conv1D
from transformers.models.llama.modeling_llama import LlamaRMSNorm

from ..import_utils import is_bnb_4bit_available, is_bnb_available
from ..utils import (
    CLAMP_QUANTILE,
    COMMON_LAYERS_PATTERN,
    TRANSFORMERS_MODELS_TO_LORA_TARGET_MODULES_MAPPING,
    ModulesToSaveWrapper,
    PeftConfig,
    PeftType,
    _freeze_adapter,
    _get_submodules,
    transpose,
)

#SGY_Modifyer
import numpy as np
gamma_value = []

if is_bnb_available():
    import bitsandbytes as bnb

@dataclass
class LoraConfig(PeftConfig):
    """
    This is the configuration class to store the configuration of a [`LoraModel`].

    Args:
        r (`int`): Lora attention dimension.
        target_modules (`Union[List[str],str]`): The names of the modules to apply Lora to.
        lora_alpha (`int`): The alpha parameter for Lora scaling.
        lora_dropout (`float`): The dropout probability for Lora layers.
        fan_in_fan_out (`bool`): Set this to True if the layer to replace stores weight like (fan_in, fan_out).
        For example, gpt-2 uses `Conv1D` which stores weights like (fan_in, fan_out) and hence this should be set to `True`.:
        bias (`str`): Bias type for Lora. Can be 'none', 'all' or 'lora_only'
        modules_to_save (`List[str]`):List of modules apart from LoRA layers to be set as trainable
            and saved in the final checkpoint.
        layers_to_transform (`Union[List[int],int]`):
            The layer indexes to transform, if this argument is specified, it will apply the LoRA transformations on
            the layer indexes that are specified in this list. If a single integer is passed, it will apply the LoRA
            transformations on the layer at this index.
        layers_pattern (`str`):
            The layer pattern name, used only if `layers_to_transform` is different from `None` and if the layer
            pattern is not in the common layers pattern.
    """
    #SGY_Modifyer
    drop_prob: float = field(default=0.0, metadata={"help": "Dropout probability for Lora layers"})
    whether_localization: bool = field(default=False, metadata={"help": "Whether to use localization"})
    interval: int = field(default=150, metadata={"help": "Interval for training lora layers"})
    lora_gamma_interval: int = field(default=50, metadata={"help": "Interval for training lora gamma layers"})
    upperbound: float = field(default=1.0, metadata={"help": "Upperbound for lora layers"})
    # fixed_layers: Optional[List[str]] = field(default=None, metadata={"help": "List of fixed layers"})

    r: int = field(default=8, metadata={"help": "Lora attention dimension"})
    target_modules: Optional[Union[List[str], str]] = field(
        default=None,
        metadata={
            "help": "List of module names or regex expression of the module names to replace with Lora."
            "For example, ['q', 'v'] or '.*decoder.*(SelfAttention|EncDecAttention).*(q|v)$' "
        },
    )
    lora_alpha: int = field(default=8, metadata={"help": "Lora alpha"})
    lora_dropout: float = field(default=0.0, metadata={"help": "Lora dropout"})
    fan_in_fan_out: bool = field(
        default=False,
        metadata={"help": "Set this to True if the layer to replace stores weight like (fan_in, fan_out)"},
    )
    bias: str = field(default="none", metadata={"help": "Bias type for Lora. Can be 'none', 'all' or 'lora_only'"})
    modules_to_save: Optional[List[str]] = field(
        default=None,
        metadata={
            "help": "List of modules apart from LoRA layers to be set as trainable and saved in the final checkpoint. "
            "For example, in Sequence Classification or Token Classification tasks, "
            "the final layer `classifier/score` are randomly initialized and as such need to be trainable and saved."
        },
    )
    init_lora_weights: bool = field(
        default=True,
        metadata={
            "help": (
                "Whether to initialize the weights of the Lora layers with their default initialization. Don't change "
                "this setting, except if you know exactly what you're doing."
            ),
        },
    )
    layers_to_transform: Optional[Union[List, int]] = field(
        default=None,
        metadata={
            "help": "The layer indexes to transform, is this argument is specified, PEFT will transform only the layers indexes that are specified inside this list. If a single integer is passed, PEFT will transform only the layer at this index."
        },
    )
    layers_pattern: Optional[str] = field(
        default=None,
        metadata={
            "help": "The layer pattern name, used only if `layers_to_transform` is different to None and if the layer pattern is not in the common layers pattern."
        },
    )
    default_localization: bool = field(
        default=False,
        metadata={
            "help": "Whether to use default localization. If True, the localization will be applied to all the layers."
        },
    )
    start_iter: int = field(default=1, metadata={"help": "Interval for calculating statistics"})
    use_sigmoid: bool = field(default=False, metadata={"help": "Whether to use sigmoid for lora layers"})
    sigmoid_init_value: float = field(default=5.0, metadata={"help": "Initial value for sigmoid"})
    block_wise: bool = field(default=False, metadata={"help": "Whether to use block-wise localization"})
    ratio: float = field(default=1.0, metadata={"help": "Ratio for Norm lora layers"})
    add_weight: float = field(default=False, metadata={"help": "Whether to add weight for lora layers"})

    def __post_init__(self):
        self.peft_type = PeftType.LORA


class LoraModel(torch.nn.Module):
    """
    Creates Low Rank Adapter (Lora) model from a pretrained transformers model.

    Args:
        model ([`~transformers.PreTrainedModel`]): The model to be adapted.
        config ([`LoraConfig`]): The configuration of the Lora model.

    Returns:
        `torch.nn.Module`: The Lora model.

    Example:

        ```py
        >>> from transformers import AutoModelForSeq2SeqLM, LoraConfig
        >>> from peft import LoraModel, LoraConfig

        >>> config = LoraConfig(
        ...     peft_type="LORA",
        ...     task_type="SEQ_2_SEQ_LM",
        ...     r=8,
        ...     lora_alpha=32,
        ...     target_modules=["q", "v"],
        ...     lora_dropout=0.01,
        ... )

        >>> model = AutoModelForSeq2SeqLM.from_pretrained("t5-base")
        >>> lora_model = LoraModel(config, model)
        ```

        ```py
        >>> import transformers
        >>> from peft import LoraConfig, PeftModel, get_peft_model, prepare_model_for_int8_training

        >>> target_modules = ["q_proj", "k_proj", "v_proj", "out_proj", "fc_in", "fc_out", "wte"]
        >>> config = LoraConfig(
        ...     r=4, lora_alpha=16, target_modules=target_modules, lora_dropout=0.1, bias="none", task_type="CAUSAL_LM"
        ... )

        >>> model = transformers.GPTJForCausalLM.from_pretrained(
        ...     "kakaobrain/kogpt",
        ...     revision="KoGPT6B-ryan1.5b-float16",  # or float32 version: revision=KoGPT6B-ryan1.5b
        ...     pad_token_id=tokenizer.eos_token_id,
        ...     use_cache=False,
        ...     device_map={"": rank},
        ...     torch_dtype=torch.float16,
        ...     load_in_8bit=True,
        ... )
        >>> model = prepare_model_for_int8_training(model)
        >>> lora_model = get_peft_model(model, config)
        ```

    **Attributes**:
        - **model** ([`~transformers.PreTrainedModel`]) -- The model to be adapted.
        - **peft_config** ([`LoraConfig`]): The configuration of the Lora model.
    """

    def __init__(self, model, config, adapter_name):
        super().__init__()
        self.model = model
        self.forward = self.model.forward
        self.peft_config = config
        self.add_adapter(adapter_name, self.peft_config[adapter_name])

        # transformers models have a .config attribute, whose presence is assumed later on
        if not hasattr(self, "config"):
            self.config = {"model_type": "custom"}

    def add_adapter(self, adapter_name, config=None):
        if config is not None:
            model_config = getattr(self.model, "config", {"model_type": "custom"})
            if hasattr(model_config, "to_dict"):
                model_config = model_config.to_dict()

            config = self._prepare_lora_config(config, model_config)
            self.peft_config[adapter_name] = config
        self._find_and_replace(adapter_name)
        if len(self.peft_config) > 1 and self.peft_config[adapter_name].bias != "none":
            raise ValueError(
                "LoraModel supports only 1 adapter with bias. When using multiple adapters, set bias to 'none' for all adapters."
            )
        mark_only_lora_as_trainable(self.model, self.peft_config[adapter_name].bias)
        if self.peft_config[adapter_name].inference_mode:
            _freeze_adapter(self.model, adapter_name)

    def _check_quantization_dependency(self):
        loaded_in_4bit = getattr(self.model, "is_loaded_in_4bit", False)
        loaded_in_8bit = getattr(self.model, "is_loaded_in_8bit", False)
        if (loaded_in_4bit or loaded_in_8bit) and not is_bnb_available():
            raise ImportError(
                "To use Lora with 8-bit or 4-bit quantization, please install the `bitsandbytes` package. "
                "You can install it with `pip install bitsandbytes`."
            )

    def _check_target_module_exists(self, lora_config, key):
        """
        check whether the module name (key) exists
        :param lora_config:
        :param key:
        :return: True / False, i.e. whether the module name (key) exists
        """
        if isinstance(lora_config.target_modules, str):
            target_module_found = re.fullmatch(lora_config.target_modules, key)
        else:
            target_module_found = any(key.endswith(target_key) for target_key in lora_config.target_modules)
            is_using_layer_indexes = getattr(lora_config, "layers_to_transform", None) is not None
            layer_indexing_pattern = getattr(lora_config, "layers_pattern", None)

            if is_using_layer_indexes and target_module_found:
                layers_pattern = COMMON_LAYERS_PATTERN if layer_indexing_pattern is None else layer_indexing_pattern
                layers_pattern = [layers_pattern] if isinstance(layers_pattern, str) else layers_pattern

                for pattern in layers_pattern:
                    layer_index = re.match(f".*.{pattern}\.(\d+)\.*", key)
                    if layer_index is not None:
                        layer_index = int(layer_index.group(1))
                        if isinstance(lora_config.layers_to_transform, int):
                            target_module_found = layer_index == lora_config.layers_to_transform
                        else:
                            target_module_found = layer_index in lora_config.layers_to_transform

                        break
                    else:
                        target_module_found = False
        return target_module_found

    def _create_new_module(self, lora_config, adapter_name, target):
        bias = hasattr(target, "bias") and target.bias is not None
        kwargs = {
            "default_localization": lora_config.default_localization,
            "drop_prob": lora_config.drop_prob,
            "whether_localization": lora_config.whether_localization,
            "interval": lora_config.interval,
            "lora_gamma_interval": lora_config.lora_gamma_interval,
            "upperbound": lora_config.upperbound,
            "r": lora_config.r,
            "ratio": lora_config.ratio,
            "lora_alpha": lora_config.lora_alpha,
            "lora_dropout": lora_config.lora_dropout,
            "fan_in_fan_out": lora_config.fan_in_fan_out,
            "init_lora_weights": lora_config.init_lora_weights,
            "start_iter": lora_config.start_iter,
            "use_sigmoid": lora_config.use_sigmoid,
            "sigmoid_init_value": lora_config.sigmoid_init_value,
            "block_wise": lora_config.block_wise,
            "add_weight": lora_config.add_weight,
        }
        loaded_in_4bit = getattr(self.model, "is_loaded_in_4bit", False)
        loaded_in_8bit = getattr(self.model, "is_loaded_in_8bit", False)

        if loaded_in_8bit and isinstance(target, bnb.nn.Linear8bitLt):
            eightbit_kwargs = kwargs.copy()
            eightbit_kwargs.update(
                {
                    "has_fp16_weights": target.state.has_fp16_weights,
                    "memory_efficient_backward": target.state.memory_efficient_backward,
                    "threshold": target.state.threshold,
                    "index": target.index,
                }
            )
            new_module = Linear8bitLt(
                adapter_name, target.in_features, target.out_features, bias=bias, **eightbit_kwargs
            )
        elif loaded_in_4bit and is_bnb_4bit_available() and isinstance(target, bnb.nn.Linear4bit):
            fourbit_kwargs = kwargs.copy()
            fourbit_kwargs.update(
                {
                    "compute_dtype": target.compute_dtype,
                    "compress_statistics": target.weight.compress_statistics,
                    "quant_type": target.weight.quant_type,
                }
            )
            new_module = Linear4bit(adapter_name, target.in_features, target.out_features, bias=bias, **fourbit_kwargs)
        elif isinstance(target, torch.nn.Embedding):
            embedding_kwargs = kwargs.copy()
            embedding_kwargs.pop("fan_in_fan_out", None)
            embedding_kwargs.pop("default_localization", None)
            embedding_kwargs.pop("drop_prob", None)
            embedding_kwargs.pop("whether_localization", None)
            embedding_kwargs.pop("interval", None)
            embedding_kwargs.pop("lora_gamma_interval", None)
            embedding_kwargs.pop("upperbound", None)
            embedding_kwargs.pop("start_iter", None)
            embedding_kwargs.pop("use_sigmoid", None)
            embedding_kwargs.pop("sigmoid_init_value", None)
            embedding_kwargs.pop("block_wise", None)
            embedding_kwargs.pop("ratio", None)
            embedding_kwargs.pop("add_weight", None)
            in_features, out_features = target.num_embeddings, target.embedding_dim
            new_module = Embedding(adapter_name, in_features, out_features, **embedding_kwargs)
        elif isinstance(target, torch.nn.Conv2d):
            out_channels, in_channels = target.weight.size()[:2]
            kernel_size = target.weight.size()[2:]
            stride = target.stride
            padding = target.padding
            new_module = Conv2d(adapter_name, in_channels, out_channels, kernel_size, stride, padding, **kwargs)
        elif isinstance(target, LlamaRMSNorm):
            hidden_size = target.weight.size()[0]
            new_module = LoRA_LlamaRMSNorm(hidden_size=hidden_size, adapter_name=adapter_name, **kwargs)
        else:
            if isinstance(target, torch.nn.Linear):
                in_features, out_features = target.in_features, target.out_features
                if kwargs["fan_in_fan_out"]:
                    warnings.warn(
                        "fan_in_fan_out is set to True but the target module is `torch.nn.Linear`. "
                        "Setting fan_in_fan_out to False."
                    )
                    kwargs["fan_in_fan_out"] = lora_config.fan_in_fan_out = False
            elif isinstance(target, Conv1D):
                in_features, out_features = (
                    target.weight.ds_shape if hasattr(target.weight, "ds_shape") else target.weight.shape
                )
                kwargs["is_target_conv_1d_layer"] = True
                if not kwargs["fan_in_fan_out"]:
                    warnings.warn(
                        "fan_in_fan_out is set to False but the target module is `Conv1D`. "
                        "Setting fan_in_fan_out to True."
                    )
                    kwargs["fan_in_fan_out"] = lora_config.fan_in_fan_out = True
            else:
                raise ValueError(
                    f"Target module {target} is not supported. "
                    f"Currently, only `torch.nn.Linear` and `Conv1D` are supported."
                )
            new_module = Linear(adapter_name, in_features, out_features, bias=bias, **kwargs)

        return new_module

    def _find_and_replace(self, adapter_name):
        lora_config = self.peft_config[adapter_name]
        self._check_quantization_dependency()
        is_target_modules_in_base_model = False
        key_list = [key for key, _ in self.model.named_modules()]

        for key in key_list:
            if not self._check_target_module_exists(lora_config, key):
                continue

            is_target_modules_in_base_model = True
            parent, target, target_name = _get_submodules(self.model, key)

            if isinstance(target, LoraLayer) and isinstance(target, torch.nn.Conv2d):
                target.update_layer_conv2d(
                    adapter_name,
                    lora_config.r,
                    lora_config.lora_alpha,
                    lora_config.lora_dropout,
                    lora_config.init_lora_weights,
                )
            elif isinstance(target, LoraLayer) and isinstance(target, torch.nn.Embedding):
                target.update_layer_embedding(
                    adapter_name,
                    lora_config.r,
                    lora_config.lora_alpha,
                    lora_config.lora_dropout,
                    lora_config.init_lora_weights,
                )

            elif isinstance(target, LoraLayer):
                target.update_layer(
                    adapter_name,
                    lora_config.r,
                    lora_config.lora_alpha,
                    lora_config.lora_dropout,
                    lora_config.init_lora_weights,
                )
            else:
                new_module = self._create_new_module(lora_config, adapter_name, target)
                self._replace_module(parent, target_name, new_module, target)

        if not is_target_modules_in_base_model:
            raise ValueError(
                f"Target modules {lora_config.target_modules} not found in the base model. "
                f"Please check the target modules and try again."
            )

    def _replace_module(self, parent_module, child_name, new_module, old_module):
        setattr(parent_module, child_name, new_module)
        new_module.weight = old_module.weight
        if hasattr(old_module, "bias"):
            if old_module.bias is not None:
                new_module.bias = old_module.bias

        if getattr(old_module, "state", None) is not None:
            new_module.state = old_module.state
            new_module.to(old_module.weight.device)

        # dispatch to correct device
        for name, module in new_module.named_modules():
            if "lora_" in name:
                module.to(old_module.weight.device)
            if "ranknum" in name:
                module.to(old_module.weight.device)

    def __getattr__(self, name: str):
        """Forward missing attributes to the wrapped module."""
        try:
            return super().__getattr__(name)  # defer to nn.Module's logic
        except AttributeError:
            return getattr(self.model, name)

    def get_peft_config_as_dict(self, inference: bool = False):
        config_dict = {}
        for key, value in self.peft_config.items():
            config = {k: v.value if isinstance(v, Enum) else v for k, v in asdict(value).items()}
            if inference:
                config["inference_mode"] = True
        config_dict[key] = config
        return config

    def _set_adapter_layers(self, enabled=True):
        for module in self.model.modules():
            if isinstance(module, LoraLayer):
                module.disable_adapters = False if enabled else True

    def enable_adapter_layers(self):
        self._set_adapter_layers(enabled=True)

    def disable_adapter_layers(self):
        self._set_adapter_layers(enabled=False)

    def set_adapter(self, adapter_name):
        for module in self.model.modules():
            if isinstance(module, LoraLayer):
                if module.merged:
                    warnings.warn("Adapter cannot be set when the model is merged. Unmerging the model first.")
                    module.unmerge()
                module.active_adapter = adapter_name

    def merge_adapter(self):
        """
        This method merges the LoRa layers into the base model.
        """
        for module in self.model.modules():
            if isinstance(module, LoraLayer):
                module.merge()

    def unmerge_adapter(self):
        """
        This method unmerges the LoRa layers from the base model.
        """
        for module in self.model.modules():
            if isinstance(module, LoraLayer):
                module.unmerge()

    @staticmethod
    def _prepare_lora_config(peft_config, model_config):
        if peft_config.target_modules is None:
            if model_config["model_type"] not in TRANSFORMERS_MODELS_TO_LORA_TARGET_MODULES_MAPPING:
                raise ValueError("Please specify `target_modules` in `peft_config`")
            peft_config.target_modules = TRANSFORMERS_MODELS_TO_LORA_TARGET_MODULES_MAPPING[model_config["model_type"]]
        return peft_config

    def _unload_and_optionally_merge(self, merge=True):
        if getattr(self.model, "is_loaded_in_8bit", False) or getattr(self.model, "is_loaded_in_4bit", False):
            raise ValueError("Cannot merge LORA layers when the model is loaded in 8-bit mode")

        key_list = [key for key, _ in self.model.named_modules() if "lora" not in key]
        for key in key_list:
            try:
                parent, target, target_name = _get_submodules(self.model, key)
            except AttributeError:
                continue
            if isinstance(target, LoraLayer):
                if isinstance(target, nn.Embedding):
                    new_module = torch.nn.Embedding(target.in_features, target.out_features)
                elif isinstance(target, nn.Conv2d):
                    new_module = torch.nn.Conv2d(
                        target.in_channels,
                        target.out_channels,
                        kernel_size=target.kernel_size,
                        stride=target.stride,
                        padding=target.padding,
                        dilation=target.dilation,
                    )
                else:
                    bias = target.bias is not None
                    if getattr(target, "is_target_conv_1d_layer", False):
                        new_module = Conv1D(target.out_features, target.in_features)
                    else:
                        new_module = torch.nn.Linear(target.in_features, target.out_features, bias=bias)
                if merge:
                    target.merge()
                self._replace_module(parent, target_name, new_module, target)

            # save any additional trainable modules part of `modules_to_save`
            if isinstance(target, ModulesToSaveWrapper):
                setattr(parent, target_name, target.modules_to_save[target.active_adapter])

        return self.model

    def add_weighted_adapter(self, adapters, weights, adapter_name, combination_type="svd"):
        """
        This method adds a new adapter by merging the given adapters with the given weights.

        Args:
            adapters (list): List of adapter names to be merged.
            weights (list): List of weights for each adapter.
            adapter_name (str): Name of the new adapter.
            combination_type (str): Type of merging. Can be one of [`svd`, `linear`]
        """
        if adapter_name in list(self.peft_config.keys()):
            return
        for adapter in adapters:
            if adapter not in list(self.peft_config.keys()):
                raise ValueError(f"Adapter {adapter} does not exist")

        # if there is only one adapter, we can only use linear merging
        combination_type = "linear" if len(adapters) == 1 else combination_type

        # new rank is the max of all ranks of the adapters
        unique_ranks = list({self.peft_config[adapter].r for adapter in adapters})
        if combination_type == "linear":
            if len(unique_ranks) != 1:
                raise ValueError("All adapters must have the same r value when using `linear` combination_type")
            new_rank = unique_ranks[0]
        elif combination_type == "svd":
            new_rank = max(unique_ranks)
        else:
            raise ValueError(f"Invalid combination_type: {combination_type}")

        self.peft_config[adapter_name] = replace(self.peft_config[adapters[0]], r=new_rank, lora_alpha=new_rank)
        self._find_and_replace(adapter_name)
        mark_only_lora_as_trainable(self.model, self.peft_config[adapter_name].bias)
        _freeze_adapter(self.model, adapter_name)
        key_list = [key for key, _ in self.model.named_modules() if "lora" not in key]
        for key in key_list:
            _, target, _ = _get_submodules(self.model, key)
            if isinstance(target, LoraLayer):
                if adapter_name in target.lora_A:
                    target_lora_A = target.lora_A[adapter_name].weight
                    target_lora_B = target.lora_B[adapter_name].weight
                elif adapter_name in target.lora_embedding_A:
                    target_lora_A = target.lora_embedding_A[adapter_name]
                    target_lora_B = target.lora_embedding_B[adapter_name]

                target_lora_A.data = target_lora_A.data * 0.0
                target_lora_B.data = target_lora_B.data * 0.0
                if combination_type == "linear":
                    for adapter, weight in zip(adapters, weights):
                        if adapter in target.lora_A:
                            current_adapter_lora_A = target.lora_A[adapter].weight
                            current_adapter_lora_B = target.lora_B[adapter].weight
                        elif adapter in target.lora_embedding_A:
                            current_adapter_lora_A = target.lora_embedding_A[adapter]
                            current_adapter_lora_B = target.lora_embedding_B[adapter]
                        target_lora_A.data += current_adapter_lora_A.data * weight * target.scaling[adapter]
                        target_lora_B.data += current_adapter_lora_B.data
                elif combination_type == "svd":
                    target_lora_A.data, target_lora_B.data = self._svd_weighted_adapter(
                        adapters, weights, new_rank, target, target_lora_A, target_lora_B
                    )

    def _svd_weighted_adapter(self, adapters, weights, new_rank, target, target_lora_A, target_lora_B):
        delta_weight = weights[0] * target.get_delta_weight(adapters[0])
        for adapter, weight in zip(adapters[1:], weights[1:]):
            delta_weight += weight * target.get_delta_weight(adapter)
        conv2d = isinstance(target, Conv2d)
        if conv2d:
            conv2d_1x1 = target.weight.size()[2:4] == (1, 1)
            if not conv2d_1x1:
                delta_weight = delta_weight.flatten(start_dim=1)
            else:
                delta_weight = delta_weight.squeeze()
        if target.fan_in_fan_out:
            delta_weight = delta_weight.T

        # based on https://github.com/kohya-ss/sd-scripts/blob/main/networks/svd_merge_lora.py#L114-L131
        U, S, Vh = torch.linalg.svd(delta_weight)
        U = U[:, :new_rank]
        S = S[:new_rank]
        U = U @ torch.diag(S)
        Vh = Vh[:new_rank, :]
        dist = torch.cat([U.flatten(), Vh.flatten()])
        hi_val = torch.quantile(dist, CLAMP_QUANTILE)
        low_val = -hi_val
        U = U.clamp(low_val, hi_val)
        Vh = Vh.clamp(low_val, hi_val)
        if conv2d:
            U = U.reshape(target_lora_B.data.shape)
            Vh = Vh.reshape(target_lora_A.data.shape)
        return Vh, U

    def delete_adapter(self, adapter_name):
        """
        Deletes an existing adapter.

        Args:
            adapter_name (str): Name of the adapter to be deleted.
        """
        if adapter_name not in list(self.peft_config.keys()):
            raise ValueError(f"Adapter {adapter_name} does not exist")
        del self.peft_config[adapter_name]
        key_list = [key for key, _ in self.model.named_modules() if "lora" not in key]
        for key in key_list:
            _, target, _ = _get_submodules(self.model, key)
            if isinstance(target, LoraLayer):
                for attr in [
                    "r",
                    "lora_alpha",
                    "scaling",
                    "lora_A",
                    "lora_B",
                    "lora_embedding_A",
                    "lora_embedding_B",
                    "lora_dropout",
                ]:
                    if adapter_name in getattr(target, attr):
                        getattr(target, attr).pop(adapter_name)
                if target.active_adapter == adapter_name:
                    resetting_active_adapter = list(self.peft_config.keys())[0]
                    warnings.warn(
                        f"Adapter {adapter_name} was active which is now deleted. Setting active adapter to {resetting_active_adapter}. "
                    )
                    target.active_adapter = resetting_active_adapter

    def merge_and_unload(self):
        r"""
        This method merges the LoRa layers into the base model. This is needed if someone wants to use the base model
        as a standalone model.

        Example:

        ```py
        >>> from transformers import AutoModelForCausalLM
        >>> from peft import PeftModel

        >>> base_model = AutoModelForCausalLM.from_pretrained("tiiuae/falcon-40b")
        >>> peft_model_id = "smangrul/falcon-40B-int4-peft-lora-sfttrainer-sample"
        >>> model = PeftModel.from_pretrained(base_model, peft_model_id)
        >>> merged_model = model.merge_and_unload()
        ```
        """
        return self._unload_and_optionally_merge()

    def unload(self):
        """
        Gets back the base model by removing all the lora modules without merging. This gives back the original base
        model.
        """
        return self._unload_and_optionally_merge(merge=False)


# Below code is based on https://github.com/microsoft/LoRA/blob/main/loralib/layers.py
# and modified to work with PyTorch FSDP


#  ------------------------------------------------------------------------------------------
#  Copyright (c) Microsoft Corporation. All rights reserved.
#  Licensed under the MIT License (MIT). See LICENSE in the repo root for license information.
#  ------------------------------------------------------------------------------------------


# had to adapt it for `lora_only` to work
def mark_only_lora_as_trainable(model: nn.Module, bias: str = "none") -> None:
    for n, p in model.named_parameters():
        if "lora_" not in n:
            p.requires_grad = False
    if bias == "none":
        return
    elif bias == "all":
        for n, p in model.named_parameters():
            if "bias" in n:
                p.requires_grad = True
    elif bias == "lora_only":
        for m in model.modules():
            if isinstance(m, LoraLayer) and hasattr(m, "bias") and m.bias is not None:
                m.bias.requires_grad = True
    else:
        raise NotImplementedError


class LoraLayer:
    def __init__(self, in_features: int, out_features: int, **kwargs):
        self.r = {}
        self.lora_alpha = {}
        self.scaling = {}
        self.lora_dropout = nn.ModuleDict({})
        self.lora_A = nn.ModuleDict({})
        self.lora_B = nn.ModuleDict({})
        # For Embedding layer
        self.lora_embedding_A = nn.ParameterDict({})
        self.lora_embedding_B = nn.ParameterDict({})
        # Mark the weight as unmerged
        self.merged = False
        self.disable_adapters = False
        self.in_features = in_features
        self.out_features = out_features
        self.kwargs = kwargs

    def update_layer(self, adapter_name, r, lora_alpha, lora_dropout, init_lora_weights, ratio=1.0):
        self.r[adapter_name] = r
        self.lora_alpha[adapter_name] = lora_alpha
        if lora_dropout > 0.0:
            lora_dropout_layer = nn.Dropout(p=lora_dropout)
        else:
            lora_dropout_layer = nn.Identity()

        self.lora_dropout.update(nn.ModuleDict({adapter_name: lora_dropout_layer}))
        # Actual trainable parameters
        if r > 0:
            self.lora_A.update(nn.ModuleDict({adapter_name: nn.Linear(self.in_features, r, bias=False)}))
            self.lora_B.update(nn.ModuleDict({adapter_name: nn.Linear(r, self.out_features, bias=False)}))
            self.scaling[adapter_name] = lora_alpha / (r * ratio)
        if init_lora_weights:
            self.reset_lora_parameters(adapter_name)
        self.to(self.weight.device)

    #SGY_Modifyer
    def freeze(self, adapter_name):
        if adapter_name in self.lora_A.keys():
            # initialize A the same way as the default for nn.Linear and B to zero
            self.lora_A[adapter_name].weight.requires_grad = False
            self.lora_B[adapter_name].weight.requires_grad = False

        if adapter_name in self.lora_embedding_A.keys():
            self.lora_embedding_A[adapter_name].requires_grad = False
            self.lora_embedding_B[adapter_name].requires_grad = False

        self.disable_adapters = True

    def update_layer_conv2d(self, adapter_name, r, lora_alpha, lora_dropout, init_lora_weights):
        self.r[adapter_name] = r
        self.lora_alpha[adapter_name] = lora_alpha
        if lora_dropout > 0.0:
            lora_dropout_layer = nn.Dropout(p=lora_dropout)
        else:
            lora_dropout_layer = nn.Identity()

        self.lora_dropout.update(nn.ModuleDict({adapter_name: lora_dropout_layer}))
        # Actual trainable parameters
        if r > 0:
            kernel_size = self.kwargs["kernel_size"]
            stride = self.kwargs["stride"]
            padding = self.kwargs["padding"]
            self.lora_A.update(
                nn.ModuleDict({adapter_name: nn.Conv2d(self.in_features, r, kernel_size, stride, padding, bias=False)})
            )
            self.lora_B.update(
                nn.ModuleDict({adapter_name: nn.Conv2d(r, self.out_features, (1, 1), (1, 1), bias=False)})
            )
            self.scaling[adapter_name] = lora_alpha / r
        if init_lora_weights:
            self.reset_lora_parameters(adapter_name)
        self.to(self.weight.device)

    def update_layer_embedding(self, adapter_name, r, lora_alpha, lora_dropout, init_lora_weights):
        self.r[adapter_name] = r
        self.lora_alpha[adapter_name] = lora_alpha
        if lora_dropout > 0.0:
            lora_dropout_layer = nn.Dropout(p=lora_dropout)
        else:
            lora_dropout_layer = nn.Identity()

        self.lora_dropout.update(nn.ModuleDict({adapter_name: lora_dropout_layer}))
        # Actual trainable parameters
        if r > 0:
            weight_A = torch.randn((r, self.in_features), dtype=self.weight.dtype, device=self.weight.device)
            weight_B = torch.randn((self.out_features, r), dtype=self.weight.dtype, device=self.weight.device)
            self.lora_embedding_A.update(nn.ParameterDict({adapter_name: nn.Parameter(weight_A)}))
            self.lora_embedding_B.update(nn.ParameterDict({adapter_name: nn.Parameter(weight_B)}))
            self.scaling[adapter_name] = lora_alpha / r
        if init_lora_weights:
            self.reset_lora_parameters(adapter_name)
        self.to(self.weight.device)

    def reset_lora_parameters(self, adapter_name):
        if adapter_name in self.lora_A.keys():
            # initialize A the same way as the default for nn.Linear and B to zero
            nn.init.kaiming_uniform_(self.lora_A[adapter_name].weight, a=math.sqrt(5))
            nn.init.zeros_(self.lora_B[adapter_name].weight)
        if adapter_name in self.lora_embedding_A.keys():
            # initialize a the same way as the default for nn.linear and b to zero
            nn.init.zeros_(self.lora_embedding_A[adapter_name])
            nn.init.normal_(self.lora_embedding_B[adapter_name])


class Linear(nn.Linear, LoraLayer):
    # Lora implemented in a dense layer
    def __init__(
        self,
        adapter_name: str,
        in_features: int,
        out_features: int,
        r: int = 0,
        lora_alpha: int = 1,
        lora_dropout: float = 0.0,
        fan_in_fan_out: bool = False,  # Set this to True if the layer to replace stores weight like (fan_in, fan_out)
        is_target_conv_1d_layer: bool = False,
        # SGY_Modifyer
        drop_prob: float = 0.0,
        whether_localization: bool = False,
        interval: int = 100,
        lora_gamma_interval: int = 50,
        upperbound: float = 1.0,
        default_localization: bool = False,
        start_iter: int = 1,
        use_sigmoid: bool = False,
        sigmoid_init_value: float = 5.0,
        block_wise: bool = False,
        ratio: float = 1.0,
        add_weight: bool = False,
        **kwargs,
    ):
        init_lora_weights = kwargs.pop("init_lora_weights", True)

        nn.Linear.__init__(self, in_features, out_features, **kwargs)
        LoraLayer.__init__(self, in_features=in_features, out_features=out_features)
        # Freezing the pre-trained weight matrix
        self.weight.requires_grad = False

        self.fan_in_fan_out = fan_in_fan_out
        if fan_in_fan_out:
            self.weight.data = self.weight.data.T

        nn.Linear.reset_parameters(self)
        self.update_layer(adapter_name, r, lora_alpha, lora_dropout, init_lora_weights)
        self.active_adapter = adapter_name
        self.is_target_conv_1d_layer = is_target_conv_1d_layer

        # SGY_Modifyer: Drop the current layer with drop_prob
        if torch.rand(1) < drop_prob:
            self.freeze(adapter_name)

        # SGY_Modifyer: Whether to localize the layer
        self.whether_localization = whether_localization
        self.use_sigmoid = use_sigmoid
        if not use_sigmoid:
            self.upperbound_value = 1.0
        else:
            self.upperbound_value = sigmoid_init_value

        self.current_iteration = 0
        self.trainer = None
        # self.parent = parent
        self.block_wise = block_wise

        self.default_localization = default_localization

        self.add_weight = add_weight
        if self.add_weight:
            self.lora_weight = nn.ParameterDict({adapter_name: nn.Parameter(torch.tensor([self.upperbound_value]))})
        self.seperate_training = False

        if whether_localization:
            self.init_localization(adapter_name, start_iter, interval, lora_gamma_interval, upperbound, default_localization,
                                   use_sigmoid, sigmoid_init_value)

    def init_localization(self, adapter_name='default', start_iter=-1,
                          interval=1, lora_gamma_interval=1, upperbound=1.0, default_localization=True,
                          use_sigmoid=False, sigmoid_init_value=1.0):
        self.start_iter = start_iter
        self.lora_score = 0
        self.lora_gamma_value_list = []
        self.diff_down = []
        self.diff_up = []
        self.upperbound_up_times = 0
        self.lowerbound_down_times = 0
        self.default_localization = default_localization

        self.interval = interval
        self.lora_gamma_interval = lora_gamma_interval
        self.state_enum = enum.Enum("state", ('finetune_lora', 'finetune_lora_gamma', 'start'))
        self.current_state = self.state_enum['start']
        self.upperbound = upperbound

        self.use_sigmoid = use_sigmoid

        if not use_sigmoid:
            self.upperbound_value = 1.0
        else:
            self.upperbound_value = 5.0

        self.lora_gamma = nn.ParameterDict({adapter_name: nn.Parameter(torch.tensor([sigmoid_init_value]))}).to(
            self.weight.device)
        self.lora_buffer = nn.ParameterDict(
            {adapter_name: nn.Parameter(torch.tensor([sigmoid_init_value]).requires_grad_(False))}).to(
            self.weight.device)

    def add_weight_init(self):
        self.add_weight = True
        self.lora_weight = nn.ParameterDict({'default': nn.Parameter(torch.tensor([self.upperbound_value]))})

    def freeze_LoRA_Weigth(self):
        self.lora_weight['default'].requires_grad = False

    #SGY_Modifyer
    def freeze_only_LoRA(self):
        self.disable_adapters = False
        for n, p in self.named_parameters():
            if "lora_" in n and 'lora_gamma' not in n and 'lora_buffer' not in n:
                p.requires_grad = False
    #SGY_Modifyer
    def freeze_only_LoRA_Gamma(self):
        self.disable_adapters = False
        for n, p in self.named_parameters():
            if 'lora_gamma' in n:
                p.requires_grad = False
    #SGY_Modifyer
    def unfreeze_only_LoRA_Gamma(self):
        self.disable_adapters = False
        for n, p in self.named_parameters():
            if 'lora_gamma' in n:
                p.requires_grad = True
    #SGY_Modifyer
    def unfreeze_only_LoRA(self):
        self.disable_adapters = False
        for n, p in self.named_parameters():
            if "lora_" in n and 'lora_gamma' not in n and 'lora_buffer' not in n and 'lora_weight' not in n:
                p.requires_grad = True

    def merge(self):
        if self.active_adapter not in self.lora_A.keys():
            return
        if self.merged:
            warnings.warn("Already merged. Nothing to do.")
            return
        if self.r[self.active_adapter] > 0:
            self.weight.data += self.get_delta_weight(self.active_adapter)
            self.merged = True

    def unmerge(self):
        if self.active_adapter not in self.lora_A.keys():
            return
        if not self.merged:
            warnings.warn("Already unmerged. Nothing to do.")
            return
        if self.r[self.active_adapter] > 0:
            self.weight.data -= self.get_delta_weight(self.active_adapter)
            self.merged = False

    def get_delta_weight(self, adapter):
        return (
            transpose(
                self.lora_B[adapter].weight @ self.lora_A[adapter].weight,
                self.fan_in_fan_out,
            )
            * self.scaling[adapter]
        )

    # SGY_Modifyer
    def fixed(self):
        self.disable_adapters = True
        for n, p in self.named_parameters():
            if "lora_" in n:
                p.requires_grad = False

    def compute_lora_score_v3(self, output):
        if self.whether_localization:
            if not self.use_sigmoid:
                output = output * self.lora_gamma[self.active_adapter]
            else:
                output = output * torch.sigmoid(self.lora_gamma[self.active_adapter])
            self.lora_score = self.lora_gamma[self.active_adapter].item()

        return output

    def compute_lora_score_v4(self, output):
        if self.whether_localization:
            self.current_iteration = self.trainer.state.global_step
            if self.current_iteration < self.start_iter:
                return output

            if not self.use_sigmoid:
                lowerbound = 0.0
            else:
                lowerbound = -5.0

            interval = self.lora_gamma_interval
            if self.current_iteration % interval == 0:
                if self.lora_gamma[self.active_adapter] == self.lora_buffer[self.active_adapter]:
                    output = output * self.lora_gamma[self.active_adapter]
                elif self.lora_gamma[self.active_adapter] > self.lora_buffer[self.active_adapter]:
                    if self.lora_buffer[self.active_adapter] == lowerbound:
                        self.diff_up.append((self.lora_gamma[self.active_adapter] -
                                             self.lora_buffer[self.active_adapter]).item())
                    else:
                        self.upperbound_up_times += 1

                    self.lora_gamma[self.active_adapter].data = torch.tensor([self.upperbound_value]).to(self.weight.device)
                    if self.use_sigmoid:
                        output = output * torch.sigmoid(self.lora_gamma[self.active_adapter])
                    else:
                        output = output * self.lora_gamma[self.active_adapter]
                    self.lora_buffer[self.active_adapter] = torch.tensor([self.upperbound_value]).to(self.weight.device)
                else:
                    if self.lora_buffer[self.active_adapter] == self.upperbound_value:
                        self.diff_down.append((self.lora_buffer[self.active_adapter]
                                               - self.lora_gamma[self.active_adapter]).item())
                    else:
                        self.lowerbound_down_times += 1

                    self.lora_gamma[self.active_adapter].data = torch.tensor([lowerbound]).to(self.weight.device)
                    if self.use_sigmoid:
                        output = output * torch.sigmoid(self.lora_gamma[self.active_adapter])
                    else:
                        output = output * self.lora_gamma[self.active_adapter]
                    self.lora_buffer[self.active_adapter] = torch.tensor([lowerbound]).to(self.weight.device)
                    self.lora_score += 1
            else:
                output = output * self.lora_gamma[self.active_adapter]

        return output

    def compute_lora_score(self, output):
        if self.whether_localization:
            self.current_iteration = self.trainer.state.global_step
            if self.current_iteration < self.start_iter:
                return output

            if not self.use_sigmoid:
                lowerbound = 0.0
            else:
                lowerbound = -5.0

            interval = self.lora_gamma_interval
            if self.current_iteration % interval == 0:
                if self.lora_gamma[self.active_adapter] == self.lora_buffer[self.active_adapter]:
                    output = output * self.lora_gamma[self.active_adapter]
                elif self.lora_gamma[self.active_adapter] > self.lora_buffer[self.active_adapter]:
                    if self.lora_buffer[self.active_adapter] == lowerbound:
                        self.diff_up.append((self.lora_gamma[self.active_adapter] -
                                             self.lora_buffer[self.active_adapter]).item())
                    else:
                        self.upperbound_up_times += 1

                    self.lora_gamma[self.active_adapter].data = torch.tensor([self.upperbound_value]).to(self.weight.device)
                    if self.use_sigmoid:
                        output = output * torch.sigmoid(self.lora_gamma[self.active_adapter])
                    else:
                        output = output * self.lora_gamma[self.active_adapter]
                    self.lora_buffer[self.active_adapter] = torch.tensor([self.upperbound_value]).to(self.weight.device)
                else:
                    if self.lora_buffer[self.active_adapter] == self.upperbound_value:
                        self.diff_down.append((self.lora_buffer[self.active_adapter]
                                               - self.lora_gamma[self.active_adapter]).item())
                    else:
                        self.lowerbound_down_times += 1

                    self.lora_gamma[self.active_adapter].data = torch.tensor([lowerbound]).to(self.weight.device)
                    if self.use_sigmoid:
                        output = output * torch.sigmoid(self.lora_gamma[self.active_adapter])
                    else:
                        output = output * self.lora_gamma[self.active_adapter]
                    self.lora_buffer[self.active_adapter] = torch.tensor([lowerbound]).to(self.weight.device)
                    self.lora_score += 1
            else:
                output = output * self.lora_gamma[self.active_adapter]

        return output

    def compute_lora_score_v2(self, output):
        if self.whether_localization:
            self.current_iteration = self.trainer.state.global_step
            interval = self.interval + self.lora_gamma_interval
            if self.current_iteration % interval <= self.interval:
                if self.current_state == self.state_enum['start'] or self.current_state == self.state_enum['finetune_lora_gamma']:
                    self.current_state = self.state_enum['finetune_lora']
                    self.unfreeze_only_LoRA()

                    # if len(gamma_value) > 0:
                    #     self.upperbound = sum(gamma_value) / len(gamma_value)
                    #
                    # if self.lora_gamma[self.active_adapter] < 1.0:
                    #     gamma_value.append(self.lora_gamma[self.active_adapter].item())

                    # check gamma value
                    if self.lora_gamma[self.active_adapter] < self.upperbound:
                        self.lora_gamma_value_list.append(self.lora_gamma[self.active_adapter].item())
                        self.lora_gamma[self.active_adapter].data = torch.tensor([0.0]).to(self.weight.device)
                        # If the gamma value is 0, then freeze the LoRA Layer
                        for n, p in self.named_parameters():
                            if "lora_" in n:
                                p.requires_grad = False
                        self.lora_score += 1
                    else:
                        self.lora_gamma_value_list.append(self.lora_gamma[self.active_adapter].item())
                        self.lora_gamma[self.active_adapter].data = torch.tensor([1.0]).to(self.weight.device)

                    self.freeze_only_LoRA_Gamma()

                output = output * self.lora_gamma[self.active_adapter]
            else:
                if self.current_state == self.state_enum['finetune_lora']:
                    self.current_state = self.state_enum['finetune_lora_gamma']
                    self.unfreeze_only_LoRA_Gamma()
                    self.freeze_only_LoRA()
                    self.lora_gamma[self.active_adapter].data = torch.tensor([1.0]).to(self.weight.device)

                output = output * self.lora_gamma[self.active_adapter]

        return output

    def compute_lora_score_v4(self, output):
        if self.whether_localization:
            self.current_iteration = self.trainer.state.global_step
            interval = self.interval + self.lora_gamma_interval
            if self.current_iteration % interval <= self.interval:
                if self.current_state == self.state_enum['start'] or self.current_state == self.state_enum['finetune_lora_gamma']:
                    self.current_state = self.state_enum['finetune_lora']
                    self.unfreeze_only_LoRA()

                    self.lora_gamma_value_list.append(self.lora_gamma[self.active_adapter].item())
                    # self.lora_gamma[self.active_adapter].data = torch.tensor([upperbound]).to(self.weight.device)

                    self.freeze_only_LoRA_Gamma()

            else:
                if self.current_state == self.state_enum['finetune_lora']:
                    self.current_state = self.state_enum['finetune_lora_gamma']
                    self.unfreeze_only_LoRA_Gamma()
                    self.freeze_only_LoRA()
                    self.lora_gamma[self.active_adapter].data = torch.tensor([self.upperbound_value]).to(self.weight.device)

                if self.use_sigmoid:
                    output = output * torch.sigmoid(self.lora_gamma[self.active_adapter])
                else:
                    output = output * self.lora_gamma[self.active_adapter]

        return output

    def fix_param(self):
        for n, p in self.named_parameters():
            p.requires_grad = False

    def forward(self, x: torch.Tensor, gamma=None):
        if self.whether_localization and self.block_wise:
            self.current_iteration = self.trainer.state.global_step
            if self.current_iteration > self.start_iter:
                assert gamma is not None

        # print(f"input dtype: {x.dtype}")
        previous_dtype = x.dtype
        if self.active_adapter not in self.lora_A.keys():
            return F.linear(x, transpose(self.weight, self.fan_in_fan_out), bias=self.bias)
        if self.disable_adapters:
            if self.r[self.active_adapter] > 0 and self.merged:
                self.unmerge()
                #SGY_Modifyer
                raise ValueError("Do not support merge and unmerge")
            result = F.linear(x, transpose(self.weight, self.fan_in_fan_out), bias=self.bias)
        elif self.r[self.active_adapter] > 0 and not self.merged:
            result = F.linear(x, transpose(self.weight, self.fan_in_fan_out), bias=self.bias)

            x = x.to(self.lora_A[self.active_adapter].weight.dtype)

            # result += (
            #         self.lora_B[self.active_adapter](
            #             self.lora_A[self.active_adapter](self.lora_dropout[self.active_adapter](x))
            #         )
            #         * self.scaling[self.active_adapter]
            # )

            # SGY_Modifyer
            output = self.lora_B[self.active_adapter](
                    self.lora_A[self.active_adapter](self.lora_dropout[self.active_adapter](x))
                ) * self.scaling[self.active_adapter]

            if self.block_wise:
                if self.whether_localization and self.current_iteration > self.start_iter:
                    if self.use_sigmoid:
                        output = output * torch.sigmoid(gamma)
                    else:
                        output = output * gamma
            else:
                if self.default_localization:
                    output = self.compute_lora_score(output)
                else:
                    output = self.compute_lora_score_v3(output)

            # if self.add_weight:
            #     if self.seperate_training:
            #         self.current_iteration = self.trainer.state.global_step
            #         if self.current_iteration > self.interval:
            #             self.unfreeze_only_LoRA()
            #             self.freeze_LoRA_Weigth()
            #
            #     if self.use_sigmoid:
            #         output = output * torch.sigmoid(self.lora_weight[self.active_adapter])
            #     else:
            #         if self.lora_weight[self.active_adapter] > 1.0:
            #             self.lora_weight[self.active_adapter].data = torch.tensor([1.0]).to(self.weight.device)
            #         if self.lora_weight[self.active_adapter] < 0.0:
            #             self.lora_weight[self.active_adapter].data = torch.tensor([0.0]).to(self.weight.device)
            #         output = output * self.lora_weight[self.active_adapter]

            result += output

        else:
            result = F.linear(x, transpose(self.weight, self.fan_in_fan_out), bias=self.bias)
            #SGY_Modifyer
            raise ValueError("Do not support merge and unmerge")

        result = result.to(previous_dtype)
        # print(f"output dtype: {result.dtype}")
        # SGY_Modifyer
        return result

class LoRA_LlamaRMSNorm(LlamaRMSNorm, LoraLayer):
    def __init__(self, hidden_size,
                 adapter_name: str,
                 eps=1e-6,
                 r: int = 0,
                 lora_alpha: int = 1,
                 lora_dropout: float = 0.0,
                 whether_localization: bool = False,
                 interval: int = 100,
                 lora_gamma_interval: int = 50,
                 upperbound: float = 1.0,
                 default_localization: bool = False,
                 start_iter: int = 1,
                 use_sigmoid: bool = False,
                 sigmoid_init_value: float = 5.0,
                 block_wise: bool = False,
                 ratio: float = 1.0,
                 add_weight: bool = False,
                 **kwargs,
                 ):
        """
        LlamaRMSNorm is equivalent to T5LayerNorm
        """
        init_lora_weights = kwargs.pop("init_lora_weights", True)

        LlamaRMSNorm.__init__(self, hidden_size, eps=eps)
        LoraLayer.__init__(self, in_features=hidden_size, out_features=hidden_size)
        self.weight.requires_grad = False
        self.update_layer(adapter_name, 2, lora_alpha, lora_dropout, init_lora_weights, ratio=ratio)
        self.active_adapter = adapter_name

        self.whether_localization = whether_localization
        self.use_sigmoid = use_sigmoid
        if not use_sigmoid:
            self.upperbound_value = 1.0
        else:
            self.upperbound_value = sigmoid_init_value

        self.lora_gamma = nn.ParameterDict({adapter_name: nn.Parameter(torch.tensor([self.upperbound_value]))})
        self.lora_buffer = nn.ParameterDict(
            {adapter_name: nn.Parameter(torch.tensor([self.upperbound_value]).requires_grad_(False))})

        self.current_iteration = 0
        self.trainer = None
        # self.parent = parent
        self.block_wise = block_wise

        self.default_localization = default_localization

        self.add_weight = add_weight
        if self.add_weight:
            self.lora_weight = nn.ParameterDict({adapter_name: nn.Parameter(torch.tensor([self.upperbound_value]))})

        if whether_localization:
            self.init_localization(adapter_name)

    def init_localization(self, adapter_name='default', start_iter=-1, interval=1, lora_gamma_interval=1, upperbound=1.0):
        if not self.whether_localization:
            self.whether_localization = True

        self.lora_gamma = nn.ParameterDict({adapter_name: nn.Parameter(torch.tensor([self.upperbound_value]))})
        self.lora_buffer = nn.ParameterDict(
            {adapter_name: nn.Parameter(torch.tensor([self.upperbound_value]).requires_grad_(False))})
        self.start_iter = start_iter
        self.lora_score = 0
        self.lora_gamma_value_list = []
        self.diff_down = []
        self.diff_up = []
        self.upperbound_up_times = 0
        self.lowerbound_down_times = 0

        self.interval = interval
        self.lora_gamma_interval = lora_gamma_interval
        self.state_enum = enum.Enum("state", ('finetune_lora', 'finetune_lora_gamma', 'start'))
        self.current_state = self.state_enum['start']
        self.upperbound = upperbound

    def fixed(self):
        self.disable_adapters = True
        for n, p in self.named_parameters():
            if "lora_" in n:
                p.requires_grad = False
    def fix_param(self):
        for n, p in self.named_parameters():
            p.requires_grad = False

    def compute_lora_score(self, output):
        if self.whether_localization:
            self.current_iteration = self.trainer.state.global_step
            if self.current_iteration < self.start_iter:
                return output

            if not self.use_sigmoid:
                lowerbound = 1e-8
            else:
                lowerbound = -5.0

            interval = self.lora_gamma_interval
            if self.current_iteration % interval == 0:
                if self.lora_gamma[self.active_adapter] == self.lora_buffer[self.active_adapter]:
                    output = output * self.lora_gamma[self.active_adapter]
                elif self.lora_gamma[self.active_adapter] > self.lora_buffer[self.active_adapter]:
                    if self.lora_buffer[self.active_adapter] == lowerbound:
                        self.diff_up.append((self.lora_gamma[self.active_adapter] -
                                             self.lora_buffer[self.active_adapter]).item())
                    else:
                        self.upperbound_up_times += 1

                    self.lora_gamma[self.active_adapter].data = torch.tensor([self.upperbound_value]).to(self.weight.device)
                    if self.use_sigmoid:
                        output = output * torch.sigmoid(self.lora_gamma[self.active_adapter])
                    else:
                        output = output * self.lora_gamma[self.active_adapter]
                    self.lora_buffer[self.active_adapter] = torch.tensor([self.upperbound_value]).to(self.weight.device)
                else:
                    if self.lora_buffer[self.active_adapter] == self.upperbound_value:
                        self.diff_down.append((self.lora_buffer[self.active_adapter]
                                               - self.lora_gamma[self.active_adapter]).item())
                    else:
                        self.lowerbound_down_times += 1

                    self.lora_gamma[self.active_adapter].data = torch.tensor([lowerbound]).to(self.weight.device)
                    if self.use_sigmoid:
                        output = output * torch.sigmoid(self.lora_gamma[self.active_adapter])
                    else:
                        output = output * self.lora_gamma[self.active_adapter]
                    self.lora_buffer[self.active_adapter] = torch.tensor([lowerbound]).to(self.weight.device)
                    self.lora_score += 1
            else:
                output = output * self.lora_gamma[self.active_adapter]

        return output

    def forward(self, hidden_states):
        input_dtype = hidden_states.dtype
        hidden_states = hidden_states.to(torch.float32)
        variance = hidden_states.pow(2).mean(-1, keepdim=True)
        hidden_states = hidden_states * torch.rsqrt(variance + self.variance_epsilon)


        if self.active_adapter not in self.lora_A.keys() or self.disable_adapters:
            results = self.weight * hidden_states.to(input_dtype)
        else:
            # print(f"1 hidden: {hidden_states.dtype}")
            hidden_states = hidden_states.to(input_dtype)
            # print(f"2 hidden: {hidden_states.dtype}")
            results = self.weight * hidden_states
            # print(f"2 results: {results.dtype}")
            # print(f"LORA in norm type: {self.lora_A[self.active_adapter].weight.dtype}")
            hidden_states = hidden_states.to(self.lora_A[self.active_adapter].weight.dtype)

            output = self.lora_B[self.active_adapter](
                self.lora_A[self.active_adapter](self.lora_dropout[self.active_adapter](hidden_states))
            ) * self.scaling[self.active_adapter]

            output = self.compute_lora_score(output)

            if self.add_weight:
                if self.use_sigmoid:
                    output = output * torch.sigmoid(self.lora_weight[self.active_adapter])
                else:
                    if self.lora_weight[self.active_adapter] > 1.0:
                        self.lora_weight[self.active_adapter].data = torch.tensor([1.0]).to(self.weight.device)
                    if self.lora_weight[self.active_adapter] < 0.0:
                        self.lora_weight[self.active_adapter].data = torch.tensor([0.0]).to(self.weight.device)
                    output = output * self.lora_weight[self.active_adapter]

            results += output
            results = results.to(input_dtype)

        return results

class Embedding(nn.Embedding, LoraLayer):
    # LoRA implemented in a Embedding layer
    def __init__(
        self,
        adapter_name: str,
        num_embeddings: int,
        embedding_dim: int,
        r: int = 0,
        lora_alpha: int = 1,
        lora_dropout: float = 0.0,
        **kwargs,
    ):
        init_lora_weights = kwargs.pop("init_lora_weights", True)

        nn.Embedding.__init__(self, num_embeddings, embedding_dim, **kwargs)
        LoraLayer.__init__(self, in_features=num_embeddings, out_features=embedding_dim)

        self.weight.requires_grad = False

        nn.Embedding.reset_parameters(self)
        self.update_layer_embedding(adapter_name, r, lora_alpha, lora_dropout, init_lora_weights)
        self.active_adapter = adapter_name

    def unmerge(self):
        if not self.merged:
            warnings.warn("Already unmerged. Nothing to do.")
            return
        if self.r[self.active_adapter] > 0:
            self.weight.data -= self.get_delta_weight(self.active_adapter)
            self.merged = False

    def merge(self):
        if self.merged:
            warnings.warn("Already merged. Nothing to do.")
            return
        if self.r[self.active_adapter] > 0:
            self.weight.data += self.get_delta_weight(self.active_adapter)
            self.merged = True

    def get_delta_weight(self, adapter):
        return transpose(self.lora_embedding_B[adapter] @ self.lora_embedding_A[adapter], True) * self.scaling[adapter]

    def forward(self, x: torch.Tensor):
        if self.disable_adapters:
            if self.r[self.active_adapter] > 0 and self.merged:
                self.unmerge()
            return nn.Embedding.forward(self, x)

        elif self.r[self.active_adapter] > 0 and not self.merged:
            result = nn.Embedding.forward(self, x)
            if self.r[self.active_adapter] > 0:
                after_A = F.embedding(
                    x,
                    self.lora_embedding_A[self.active_adapter].T,
                    self.padding_idx,
                    self.max_norm,
                    self.norm_type,
                    self.scale_grad_by_freq,
                    self.sparse,
                )
                result += (after_A @ self.lora_embedding_B[self.active_adapter].T) * self.scaling[self.active_adapter]
            return result
        else:
            return nn.Embedding.forward(self, x)


class Conv2d(nn.Conv2d, LoraLayer):
    # Lora implemented in a conv2d layer
    def __init__(
        self,
        adapter_name: str,
        in_channels: int,
        out_channels: int,
        kernel_size: Union[int, Tuple[int]],
        stride: Union[int, Tuple[int]] = 1,
        padding: Union[int, Tuple[int]] = 0,
        r: int = 0,
        lora_alpha: int = 1,
        lora_dropout: float = 0.0,
        **kwargs,
    ):
        init_lora_weights = kwargs.pop("init_lora_weights", True)

        nn.Conv2d.__init__(self, in_channels, out_channels, kernel_size, stride, padding)
        LoraLayer.__init__(
            self,
            in_features=in_channels,
            out_features=out_channels,
            kernel_size=kernel_size,
            stride=stride,
            padding=padding,
        )
        # Freezing the pre-trained weight matrix
        self.weight.requires_grad = False

        nn.Conv2d.reset_parameters(self)
        self.update_layer_conv2d(adapter_name, r, lora_alpha, lora_dropout, init_lora_weights)
        self.active_adapter = adapter_name

    def merge(self):
        if self.active_adapter not in self.lora_A.keys():
            return
        if self.merged:
            warnings.warn("Already merged. Nothing to do.")
            return
        if self.r[self.active_adapter] > 0:
            self.weight.data += self.get_delta_weight(self.active_adapter)
            self.merged = True

    def unmerge(self):
        if self.active_adapter not in self.lora_A.keys():
            return
        if not self.merged:
            warnings.warn("Already unmerged. Nothing to do.")
            return
        if self.r[self.active_adapter] > 0:
            self.weight.data -= self.get_delta_weight(self.active_adapter)
            self.merged = False

    def get_delta_weight(self, adapter):
        # https://github.com/bmaltais/kohya_ss/blob/feb6728762a8f463d15ba936d189d4c3abfaa1ab/networks/lora.py#L117
        if self.weight.size()[2:4] == (1, 1):
            # conv2d 1x1
            return (
                self.lora_B[adapter].weight.squeeze(3).squeeze(2) @ self.lora_A[adapter].weight.squeeze(3).squeeze(2)
            ).unsqueeze(2).unsqueeze(3) * self.scaling[adapter]
        else:
            # conv2d 3x3
            return (
                F.conv2d(
                    self.lora_A[adapter].weight.permute(1, 0, 2, 3),
                    self.lora_B[adapter].weight,
                ).permute(1, 0, 2, 3)
                * self.scaling[adapter]
            )

    def forward(self, x: torch.Tensor):
        previous_dtype = x.dtype

        if self.active_adapter not in self.lora_A.keys():
            return F.conv2d(
                x,
                self.weight,
                bias=self.bias,
                stride=self.stride,
                padding=self.padding,
                dilation=self.dilation,
                groups=self.groups,
            )
        if self.disable_adapters:
            if self.r[self.active_adapter] > 0 and self.merged:
                self.unmerge()
            result = F.conv2d(
                x,
                self.weight,
                bias=self.bias,
                stride=self.stride,
                padding=self.padding,
                dilation=self.dilation,
                groups=self.groups,
            )
        elif self.r[self.active_adapter] > 0 and not self.merged:
            result = F.conv2d(
                x,
                self.weight,
                bias=self.bias,
                stride=self.stride,
                padding=self.padding,
                dilation=self.dilation,
                groups=self.groups,
            )

            x = x.to(self.lora_A[self.active_adapter].weight.dtype)

            result += (
                self.lora_B[self.active_adapter](
                    self.lora_A[self.active_adapter](self.lora_dropout[self.active_adapter](x))
                )
                * self.scaling[self.active_adapter]
            )
        else:
            result = F.conv2d(
                x,
                self.weight,
                bias=self.bias,
                stride=self.stride,
                padding=self.padding,
                dilation=self.dilation,
                groups=self.groups,
            )

        result = result.to(previous_dtype)

        return result


if is_bnb_available():

    class Linear8bitLt(bnb.nn.Linear8bitLt, LoraLayer):
        # Lora implemented in a dense layer
        def __init__(
            self,
            adapter_name,
            in_features,
            out_features,
            r: int = 0,
            lora_alpha: int = 1,
            lora_dropout: float = 0.0,
            **kwargs,
        ):
            bnb.nn.Linear8bitLt.__init__(
                self,
                in_features,
                out_features,
                bias=kwargs.get("bias", True),
                has_fp16_weights=kwargs.get("has_fp16_weights", True),
                memory_efficient_backward=kwargs.get("memory_efficient_backward", False),
                threshold=kwargs.get("threshold", 0.0),
                index=kwargs.get("index", None),
            )
            LoraLayer.__init__(self, in_features=in_features, out_features=out_features)

            # Freezing the pre-trained weight matrix
            self.weight.requires_grad = False
            init_lora_weights = kwargs.pop("init_lora_weights", True)
            self.update_layer(adapter_name, r, lora_alpha, lora_dropout, init_lora_weights)
            self.active_adapter = adapter_name

        def forward(self, x: torch.Tensor):
            result = super().forward(x)

            if self.disable_adapters or self.active_adapter not in self.lora_A.keys():
                return result
            elif self.r[self.active_adapter] > 0:
                if not torch.is_autocast_enabled():
                    expected_dtype = result.dtype

                    if x.dtype != torch.float32:
                        x = x.float()
                    output = (
                        self.lora_B[self.active_adapter](
                            self.lora_A[self.active_adapter](self.lora_dropout[self.active_adapter](x))
                        ).to(expected_dtype)
                        * self.scaling[self.active_adapter]
                    )
                else:
                    output = (
                        self.lora_B[self.active_adapter](
                            self.lora_A[self.active_adapter](self.lora_dropout[self.active_adapter](x))
                        )
                        * self.scaling[self.active_adapter]
                    )
                result += output
            return result

    if is_bnb_4bit_available():

        class Linear4bit(bnb.nn.Linear4bit, LoraLayer):
            # Lora implemented in a dense layer
            def __init__(
                self,
                adapter_name,
                in_features,
                out_features,
                r: int = 0,
                lora_alpha: int = 1,
                lora_dropout: float = 0.0,
                # SGY_Modifyer
                drop_prob: float = 0.0,
                whether_localization: bool = False,
                **kwargs,
            ):
                bnb.nn.Linear4bit.__init__(
                    self,
                    in_features,
                    out_features,
                    bias=kwargs.get("bias", True),
                    compute_dtype=kwargs.get("compute_dtype", torch.float32),
                    compress_statistics=kwargs.get("compress_statistics", True),
                    quant_type=kwargs.get("quant_type", "nf4"),
                )
                LoraLayer.__init__(self, in_features=in_features, out_features=out_features)

                # Freezing the pre-trained weight matrix
                self.weight.requires_grad = False

                init_lora_weights = kwargs.pop("init_lora_weights", True)
                self.update_layer(adapter_name, r, lora_alpha, lora_dropout, init_lora_weights)
                self.active_adapter = adapter_name

                # SGY_Modifyer: Drop the current layer with drop_prob
                if torch.rand(1) < drop_prob:
                    self.freeze(adapter_name)

                # SGY_Modifyer: Whether to localize the layer
                self.whether_localization = whether_localization
                self.lora_gamma = nn.ParameterDict({adapter_name: nn.Parameter(torch.tensor([1.0]))})
                self.current_iteration = 0

                if whether_localization:
                    self.lora_buffer = nn.ParameterDict({adapter_name: nn.Parameter(torch.tensor([1.0]).requires_grad_(False))})
                    self.lora_score = 0
                    self.diff_down = []
                    self.diff_up = []
                    self.upperbound_up_times = 0
                    self.lowerbound_down_times = 0

            # SGY_Modifyer
            def fixed(self):
                self.disable_adapters = True
                for n, p in self.named_parameters():
                    if "lora_" in n:
                        p.requires_grad = False

            def forward(self, x: torch.Tensor):
                result = super().forward(x)

                if self.disable_adapters or self.active_adapter not in self.lora_A.keys():
                    return result
                elif self.r[self.active_adapter] > 0:
                    result = result.clone()
                    if not torch.is_autocast_enabled():
                        expected_dtype = result.dtype
                        x = x.to(self.lora_A[self.active_adapter].weight.dtype)
                        output = (
                            self.lora_B[self.active_adapter](
                                self.lora_A[self.active_adapter](self.lora_dropout[self.active_adapter](x))
                            ).to(expected_dtype)
                            * self.scaling[self.active_adapter]
                        )
                    else:
                        output = (
                            self.lora_B[self.active_adapter](
                                self.lora_A[self.active_adapter](self.lora_dropout[self.active_adapter](x))
                            )
                            * self.scaling[self.active_adapter]
                        )
                    #SGY_Modifyer
                    if self.whether_localization and self.current_iteration > 100:
                        lowerbound = 1e-8
                        if self.lora_gamma[self.active_adapter] == self.lora_buffer[self.active_adapter]:
                            output = output * self.lora_gamma[self.active_adapter]
                        elif self.lora_gamma[self.active_adapter] > self.lora_buffer[self.active_adapter]:
                            if self.lora_buffer[self.active_adapter] == lowerbound:
                                self.diff_up.append((self.lora_gamma[self.active_adapter] -
                                                     self.lora_buffer[self.active_adapter]).item())
                            else:
                                self.upperbound_up_times += 1

                            self.lora_gamma[self.active_adapter].data = torch.tensor([1.0]).to(self.weight.device)
                            output = output * self.lora_gamma[self.active_adapter]
                            self.lora_buffer[self.active_adapter] = torch.tensor([1.0]).to(self.weight.device)
                        else:
                            if self.lora_buffer[self.active_adapter] == 1.0:
                                self.diff_down.append((self.lora_buffer[self.active_adapter]
                                                       -self.lora_gamma[self.active_adapter]).item())
                            else:
                                self.lowerbound_down_times += 1

                            self.lora_gamma[self.active_adapter].data = torch.tensor([lowerbound]).to(self.weight.device)
                            output = output * self.lora_gamma[self.active_adapter]
                            self.lora_buffer[self.active_adapter] = torch.tensor([lowerbound]).to(self.weight.device)
                            self.lora_score += 1

                    result += output
                #SGY_Modifyer
                self.current_iteration += 1
                return result
