import numpy as np

class Categorical:

    def __init__(self, probabilities, initialize_log=False):
        if initialize_log:
            self._p = np.exp(probabilities)
            self._logp = probabilities
        else:
            self._p = probabilities
            self._logp = np.log(probabilities)

    def sample(self, num_samples):
        thresholds = np.expand_dims(np.cumsum(self._p), 0)
        thresholds[0, -1] = 1.0
        eps = np.random.uniform(size=[num_samples, 1])
        samples = np.argmax(eps < thresholds, axis=-1)
      
        return samples

    @property
    def probabilities(self):
        return self._p

    @probabilities.setter
    def probabilities(self, new_probabilities):
        self._p = new_probabilities
        self._logp = np.log(new_probabilities)

    @probabilities.setter
    def log_probabilities(self, new_log_probabilities):
        self._p = np.exp(new_log_probabilities)
        self._logp = new_log_probabilities

    @property
    def log_probabilities(self):
        return self._logp

    def add_entry(self, new_prob):
        p = np.concatenate([self._p, new_prob * np.ones(1,)], axis=0)
        self._p = p / np.sum(p)

    def remove_entry(self, idx):
        if idx > self._p.size:
            raise AssertionError("Invalid Index for Categorical")
        p = np.concatenate([self._p[:idx], self._p[idx + 1:]], axis=0)
        self._p = p / np.sum(p)

    def entropy(self):
        return - np.sum(self._p * self.log_probabilities)

    def kl(self, other):
        return np.sum(self._p * (self.log_probabilities - other.log_probabilities))

if __name__ == "__main__":

    p = np.array([0.3, 0.2, 0.1, 0.4])

    cat = Categorical(p)

    s = cat.sample(100000000)
    for i in range(4):
        print(i, np.count_nonzero(s == i) / 100000000)