import numpy as np

class RingBuffer:

    def __init__(self, buf_size, num_initial_entries, dtype=np.float64, default_val=-1):
        """
        Implements ring buffer with adaptable width per slot
        :param buf_size: number of sluts
        :param num_initial_entries: initial number of entries per slot
        :param dtype: data type of slots
        :param default_val: default value, should not occur be written to buffer
        """
        self._buf_size = buf_size
        self._default_val = default_val
        self._dtype = dtype
        self._cur_idx = 0
        self._buf = np.ones((num_initial_entries, self._buf_size), dtype=self._dtype) * self._default_val

    def write(self, data):
        """
        writes data to current slot
        :param data:
        """
        self._buf[:, self._cur_idx % self._buf_size] = data
        self._cur_idx += 1

    def add_entry(self, new_idx):
        """
        Increases width of slots by one, initialized with default value
        :param new_idx: index of new slot
        :return:
        """
        new_row = np.ones((1, self._buf_size), dtype=self._dtype) * self._default_val
        self._buf = np.concatenate((self._buf[:new_idx], new_row, self._buf[new_idx:]), 0)

    def del_entry(self, entry):
        """
        Decreases width of slots by one, by deleting entry number "entry"
        :param entry:
        :return:
        """
        self._buf = np.concatenate((self._buf[:entry], self._buf[(entry + 1):]), 0)

    @property
    def written(self):
        """
        :return: Array, indicating if a entry is written in all slots
        """
        return np.logical_not(np.any(self._buf == self._default_val, axis=1))

    @property
    def data(self):
        """
        raw array containing current buffer data
        :return:
        """
        return self._buf

