from pathlib import Path

from lita.core.tools import StrReplaceEditor
from lita.core.utils import read_file

from .tools import CodeActTool

_DETAILED_STR_REPLACE_EDITOR_DESCRIPTION = """Custom editing tool for viewing, \
creating and editing files in plain-text format
* State is persistent across command calls and discussions with the user
* If `path` is a text file, `view` displays the result of applying `cat -n`. \
If `path` is a directory, `view` lists non-hidden files and directories up to 2 levels deep
* The following binary file extensions can be viewed in Markdown format: \
[".xlsx", ".pptx", ".wav", ".mp3", ".m4a", ".flac", ".pdf", ".docx"]. IT DOES NOT HANDLE IMAGES.
* The `create` command cannot be used if the specified `path` already exists as a file
* If a `command` generates a long output, it will be truncated and marked with `<response clipped>`
* The `undo_edit` command will revert the last edit made to the file at `path`
* This tool can be used for creating and editing files in plain-text format.


Before using this tool:
1. Use the view tool to understand the file's contents and context
2. Verify the directory path is correct (only applicable when creating new files):
   - Use the view tool to verify the parent directory exists and is the correct location

When making edits:
   - Ensure the edit results in idiomatic, correct code
   - Do not leave the code in a broken state
   - Always use absolute file paths (starting with /)

CRITICAL REQUIREMENTS FOR USING THIS TOOL:

1. EXACT MATCHING: The `old_str` parameter must match EXACTLY one or more consecutive lines from the file, \
including all whitespace and indentation. \
The tool will fail if `old_str` matches multiple locations or doesn't match exactly with the file content.

2. UNIQUENESS: The `old_str` must uniquely identify a single instance in the file:
   - Include sufficient context before and after the change point (3-5 lines recommended)
   - If not unique, the replacement will not be performed

3. REPLACEMENT: The `new_str` parameter should contain the edited lines that replace the `old_str`. \
Both strings must be different.

Remember: when making multiple file edits in a row to the same file, \
you should prefer to send all edits in a single message with multiple calls to this tool, \
rather than multiple messages with a single call each.
"""

_SHORT_STR_REPLACE_EDITOR_DESCRIPTION = """Custom editing tool for viewing, \
creating and editing files in plain-text format
* State is persistent across command calls and discussions with the user
* If `path` is a file, `view` displays the result of applying `cat -n`. \
If `path` is a directory, `view` lists non-hidden files and directories up to 2 levels deep
* The `create` command cannot be used if the specified `path` already exists as a file
* If a `command` generates a long output, it will be truncated and marked with `<response clipped>`
* The `undo_edit` command will revert the last edit made to the file at `path`
Notes for using the `str_replace` command:
* The `old_str` parameter should match EXACTLY one or more consecutive lines from the original file. \
Be mindful of whitespaces!
* If the `old_str` parameter is not unique in the file, the replacement will not be performed. \
Make sure to include enough context in `old_str` to make it unique
* The `new_str` parameter should contain the edited lines that should replace the `old_str`
"""


class CAStrReplaceEditor(CodeActTool, StrReplaceEditor):
    name: str = "str_replace_editor"
    description: str = _DETAILED_STR_REPLACE_EDITOR_DESCRIPTION
    parameters: dict = {
        "type": "object",
        "properties": {
            'command': {
                'description': 'The commands to run. Allowed options are: '
                + '`view`, `create`, `str_replace`, `insert`, `undo_edit`.',
                'enum': [
                    'view',
                    'create',
                    'str_replace',
                    'insert',
                    'undo_edit',
                ],
                'type': 'string',
            },
            'path': {
                'description': 'Absolute path to file or directory, ' + 'e.g. `/workspace/file.py` or `/workspace`.',
                'type': 'string',
            },
            'file_text': {
                'description': 'Required parameter of `create` command, with the content of the file to be created.',
                'type': 'string',
            },
            'old_str': {
                'description': 'Required parameter of `str_replace` command '
                + 'containing the string in `path` to replace.',
                'type': 'string',
            },
            'new_str': {
                'description': 'Optional parameter of `str_replace` command containing the new string '
                + '(if not given, no string will be added). '
                + 'Required parameter of `insert` command containing the string to insert.',
                'type': 'string',
            },
            'insert_line': {
                'description': 'Required parameter of `insert` command. '
                + 'The `new_str` will be inserted AFTER the line `insert_line` of `path`.',
                'type': 'integer',
            },
            'view_range': {
                'description': 'Optional parameter of `view` command when `path` points to a file. '
                + 'If none is given, the full file is shown. '
                + 'If provided, the file will be shown in the indicated line number range, '
                + 'e.g. [11, 12] will show lines 11 and 12. Indexing at 1 to start. '
                + 'Setting `[start_line, -1]` shows all lines from `start_line` to the end of the file.',
                'items': {'type': 'integer'},
                'type': 'array',
            },
        },
        "required": ["path", "command"],
    }

    def __init__(self, **kwargs):
        use_short_description: bool = kwargs.pop("use_short_description", False)
        super().__init__(**kwargs)
        if use_short_description:
            self.description = _SHORT_STR_REPLACE_EDITOR_DESCRIPTION

    def insert(self, path: str, new_str: str, insert_line: int) -> str:
        _path = Path(path)
        if not _path.exists():
            return f"{self.name}: File not found: {path}"
        if not _path.is_file():
            return f"{self.name}: `insert` operation only supports text files, but got: {path}"

        try:
            lines = read_file(_path, splitlines=True)
            if not lines:
                return f"{self.name}: File is empty: {path}"

            if insert_line < 1 or insert_line > len(lines):
                return f"{self.name}: Invalid `insert_line`: {insert_line}."

            # Insert after the line, remember to include `\n`
            lines.insert(insert_line, new_str + "\n")

            _path.write_text("".join(lines), encoding="utf-8")
            return "New content inserted."
        except Exception as e:
            return f"{self.name}: Error editing file: {e}"

    def list_dir_2_levels(self, path: str) -> str:
        if Path(path).exists():
            # List non-hidden files and directories up to 2 levels deep
            # Use `glob` instead of `rglob` to avoid too deep files
            files = []
            p = Path(path)
            for entry in p.glob("*"):
                if entry.name.startswith("."):
                    continue
                if entry.is_file() or entry.is_dir():
                    files.append(entry.relative_to(p))
                if entry.is_dir():
                    for subentry in entry.glob("*"):
                        if subentry.name.startswith("."):
                            continue
                        if subentry.is_file() or subentry.is_dir():
                            files.append(subentry.relative_to(p))
            files = [str(f) for f in files if len(f.parts) <= 2]
            if not files:
                return "Directory is empty."
            return "\n".join(files)
        else:
            return f"{self.name}: Path not found: {path}"

    # Overwrite
    def view(self, path: str, view_range: list[int] = None) -> str:
        if Path(path).is_file():
            return super().view(path, view_range)
        else:
            return self.list_dir_2_levels(path)

    def execute(self, **kwargs) -> str:
        operation = kwargs.get("command")
        path = kwargs.get("path")

        if operation == "view":
            view_range = kwargs.get("view_range")
            return self.view(path, view_range)

        elif operation == "create":
            file_text = kwargs.get("file_text")
            if not file_text:
                return f"{self.name}: Please specify `content` in `create` operation."
            return self.create(path, file_text)

        elif operation == "insert":
            new_str = kwargs.get("new_str")
            insert_line = kwargs.get("insert_line")
            if not insert_line or not new_str:
                return f"{self.name}: Please specify both `insert_line` and `new_str` in `insert` operation."
            return self.insert(path, new_str, insert_line)

        elif operation == "str_replace":
            old_str = kwargs.get("old_str")
            new_str = kwargs.get("new_str")
            if not old_str:
                return f"{self.name}: Please specify `old_str` in `str_replace` operation."
            if new_str is None:
                # NOTE: can be empty string here
                return f"{self.name}: Please specify `new_str` in `str_replace` operation."
            return self.edit(path, old_str, new_str)

        elif operation == "undo_edit":
            return "`undo_edit` has not been implemented by the system now. Please try other ways."

        else:
            return f"{self.name}: {operation} is unsupported."
