from pathlib import Path

from lita.core.tools import Tool
from lita.core.utils import apply_diff, read_file


class Planning(Tool):
    name: str = "planning"
    description: str = "This planning tool helps you view, create, update, and manage plans for complex tasks."

    parameters: dict = {
        "type": "object",
        "properties": {
            "operation": {
                "description": "The operation to perform.",
                "enum": [
                    "view",
                    "create",
                    "update",
                ],
                "type": "string",
            },
            "path": {
                "description": "The absolute file path for storing the plan.",
                "type": "string",
            },
            "content": {
                "description": "The Markdown-formatted overall plan to achieve the goal. Required for `create`.",
                "type": "string",
            },
            "diff": {
                "description": (
                    "Update the plan in the `path` with the SEARCH/REPLACE diff blocks. Required for `update`."
                ),
                "type": "string",
            },
        },
        "required": ["operation", "path"],
        "additionalProperties": False,
    }

    def execute(self, **kwargs) -> str:
        operation = kwargs.get("operation")
        path = kwargs.get("path")

        if not operation:
            return f"{self.name}: `operation` is required."
        if not path:
            return f"{self.name}: `path` is required."

        _path = Path(path)
        if operation == "view":
            if not _path.exists():
                return f"{self.name}: File not found: {path}"
            plan_content = _path.read_text()
            return plan_content
        elif operation == "create":
            content = kwargs.get("content")
            if content is None:
                return f"{self.name}: `content` is need for the `create`."
            _path.write_text(content)
            return "created plan."
        elif operation == "update":
            if not _path.exists():
                return f"{self.name}: File not found: {path}"
            diff = kwargs.get("diff")
            if diff is None:
                return f"{self.name}: `diff` is need for the `update`."
            # Apply the diff to the existing plan
            try:
                file_content = read_file(_path)
                stat, updated = apply_diff(file_content, diff)
                if stat:
                    _path.write_text(updated)
                    return "Updated plan."
                else:
                    # Return the error message from apply_diff
                    return updated
            except Exception as e:
                return f"{self.name}: Error editing plan: {e}"

        return f"{self.name}: {operation} is unsupported."
