from pathlib import Path

from pydantic.json_schema import SkipJsonSchema

from lita.core.terminal.terminal_local import LocalTerminal
from lita.core.terminal.terminal_local_docker import DockerTerminal
from lita.core.tools import DiffBlockEditor, StrReplaceEditor, Tool
from lita.core.utils import search_files


class LitaTool(Tool):
    name: str
    description: str
    parameters: dict


class Finish(LitaTool):
    name: str = "finish"
    description: str = "Finish the task and return a final message."
    parameters: dict = {
        "type": "object",
        "required": ["message", "status"],
        "properties": {
            "message": {
                "type": "string",
                "description": "Final message or summary of the task.",
            },
            "status": {
                "type": "string",
                "enum": ["true", "false", "partial"],
                "description": "Whether the task is completed.",
            },
        },
    }

    def execute(self, message: str, status: str) -> str:
        return "STOP THE RUN HERE."


class Think(LitaTool):
    name: str = "think"
    description: str = "Used for self-think about something."
    parameters: dict = {
        "type": "object",
        "properties": {
            "thought": {
                "type": "string",
                "description": "the thinking content.",
            }
        },
        "required": ["thought"],
    }

    def execute(self, thought: str) -> str:
        return "thinking completed."


class TerminalBash(LitaTool):
    name: str = "bash"
    description: str = "Execute a command in the terminal and return the output."
    parameters: dict = {
        "type": "object",
        "properties": {
            "command": {
                "type": "string",
                "description": "Command to execute in the bash shell.",
            },
        },
        "required": ["command"],
    }
    terminal: SkipJsonSchema[LocalTerminal | DockerTerminal]

    def execute(self, **kwargs) -> str:
        # No more checking is needed
        command = kwargs.get("command", "")

        result = self.terminal.execute(command)
        return result


class TextFileEditor(LitaTool, DiffBlockEditor):
    name: str = "text_file_editor"
    description: str = """
A tool for viewing, creating, and editing text files:
* The view command returns the contents of a file, optionally limited to a specified range of lines.
* The create command creates a new file with the provided content.
* The edit command applies diff-based edits to an existing file, sequentially replacing all blocks.

Note: The diff format is a series of SEARCH/REPLACE blocks, every block must use this format:
1. The start of search block: <<<<<<< SEARCH
2. A contiguous chunk of lines to search for in the existing file
3. The dividing line: =======
4. The lines to replace into the source file
5. The end of replace block: >>>>>>> REPLACE
""".strip()  # noqa: E501

    parameters: dict = {
        "type": "object",
        "properties": {
            "operation": {
                "description": "File operation to perform.",
                "type": "string",
                "enum": ["view", "create", "edit"],
            },
            "path": {
                "description": "Absolute file path.",
                "type": "string",
            },
            "content": {
                "description": "Plain text content. Required for `create`.",
                "type": "string",
            },
            "diff": {
                "description": "Diff content (a series of SEARCH/REPLACE blocks). Required for `edit`.",
                "type": "string",
            },
            "view_range": {
                "description": "Line range to view (1-based start and end line numbers). Optional for `view`.",
                "type": "array",
                "items": {"type": "integer"},
            },
        },
        "required": ["operation", "path"],
    }

    def execute(self, **kwargs) -> str:
        operation = kwargs.get("operation")
        path = kwargs.get("path")

        if operation == "view":
            view_range = kwargs.get("view_range")

            # json.loads('{"line_number":true}') => {"line_number": True}
            return self.view(path, view_range, False)
        elif operation == "create":
            content = kwargs.get("content")
            if content is None:
                return f"{self.name}: Please specify `content` in `create` operation."
            return self.create(path, content)
        elif operation == "edit":
            diff = kwargs.get("diff")
            if diff is None:
                return f"{self.name}: Please specify `diff` in `edit` operation."
            return self.edit(path, diff)
        else:
            return f"{self.name}: {operation} is unsupported."


class TextFileStrReplaceEditor(LitaTool, StrReplaceEditor):
    name: str = "text_file_editor"
    description: str = """
A tool for viewing, creating, and editing text files:
* The view command returns the contents of a file, optionally limited to a specified range of lines.
* The create command creates a new file with the provided content.
* edit: replaces a specific string of text `old_str` with new content `new_str` in a file.

Note:
For the edit command to work correctly, the old_str must exactly match one or more consecutive lines in the file. This includes all whitespace and indentation.
The old_str must be an exact, one-to-one match for a section of the file. If it doesn't match or matches multiple locations, the command will fail.
""".strip()  # noqa: E501

    parameters: dict = {
        "type": "object",
        "properties": {
            "operation": {
                "description": "File operation to perform.",
                "type": "string",
                "enum": ["view", "create", "edit"],
            },
            "path": {
                "description": "Absolute file path.",
                "type": "string",
            },
            "content": {
                "description": "Plain text content. Required for `create`.",
                "type": "string",
            },
            "old_str": {
                "description": "Text to replace. Required for `edit`.",
                "type": "string",
            },
            "new_str": {
                "description": "Replacement text. Required for `edit`.",
                "type": "string",
            },
            "view_range": {
                "description": "Line range to view (1-based start and end line numbers). Optional for `view`.",
                "type": "array",
                "items": {"type": "integer"},
            },
        },
        "required": ["operation", "path"],
    }

    def execute(self, **kwargs) -> str:
        operation = kwargs.get("operation")
        path = kwargs.get("path")

        if operation == "view":
            view_range = kwargs.get("view_range")

            # json.loads('{"line_number":true}') => {"line_number": True}
            return self.view(path, view_range, False)
        elif operation == "create":
            content = kwargs.get("content")
            if content is None:
                return f"{self.name}: Please specify `content` in `create` operation."
            return self.create(path, content)
        elif operation == "edit":
            old_str = kwargs.get("old_str")
            new_str = kwargs.get("new_str")
            if old_str is None:
                return f"{self.name}: Please specify `old_str` in `edit` operation."
            if new_str is None:
                return f"{self.name}: Please specify `new_str` in `edit` operation."
            return self.edit(path, old_str, new_str)
        else:
            return f"{self.name}: {operation} is unsupported."


class Search(LitaTool):
    name: str = "search"
    description: str = (
        "Search for a keyword in a folder and return the relevant files's path. "
        "Files listed in each folder's .gitignore will be excluded."
    )
    parameters: dict = {
        "type": "object",
        "properties": {
            "query": {
                "type": "string",
                "description": "Keyword to search.",
            },
            "path": {
                "type": "string",
                "description": "Absolute folder path to search in.",
            },
            "file_type": {
                "type": "string",
                "description": "Optional. File type to search for (e.g., '.txt', '.py', '.java').",
            },
        },
        "required": ["query", "path"],
    }

    def execute(self, **kwargs) -> str:
        query = kwargs.get("query", "")
        path = kwargs.get("path", "")
        file_type = kwargs.get("file_type")
        _path = Path(path)
        if not _path.exists():
            return f"Directory does not exist: {path}"

        if not _path.is_dir():
            return f"Path is not a directory: {path}"

        files = search_files(_path, query, file_type)
        if not files:
            return "No matching files found."
        return "\n".join(str(file) for file in files)
