import subprocess
import uuid
from pathlib import Path

import docker
from docker.models.containers import Container

from .terminal import Terminal


class DockerTerminal(Terminal):
    workdir: str = None
    _container: Container = None

    def __init__(self, image_name: str, workdir: str, **kwargs):
        self.container_id: str | None = None
        self.container_name: str | None = kwargs.get("container_name")

        try:
            self._client = docker.from_env()
            self.setup(
                image_name=image_name,
                workdir=workdir,
                container_name=self.container_name,
            )
        except Exception as e:
            raise RuntimeError("Error creating docker client.") from e

    def setup(self, image_name: str, workdir: str, container_name: str | None = None, **kwargs):
        self.workdir = workdir
        if not container_name:
            container_name = f"container-{uuid.uuid4().hex[:8]}"
        # --mount type=bind,source=/user-local/my_project,target=/user-local/my_project
        local_workdir = kwargs.get("local_workdir")
        if local_workdir:
            _local_workdir = Path(local_workdir).resolve()
            _workdir = Path(workdir).resolve()
            # "local_workdir must match the workdir"
            assert _local_workdir == _workdir, f"local_workdir {local_workdir} must match the workdir {workdir}"

        # docker run -d --rm --name [container_name] -v [local_workdir]:[workdir] [IMAGE_NAME] sleep [DURATION]
        cmd = ["docker", "run", "-d", "--rm"]
        if local_workdir:
            cmd.extend(["--mount", f"type=bind,source={local_workdir},target={workdir}"])

        cmd.extend(["--user", "root"])
        cmd_tail = [
            "--name",
            container_name,
            "-w",
            workdir,
            image_name,
            "sleep",
            "15m",  # the maximum duration for the container to run
        ]
        cmd.extend(cmd_tail)
        result = subprocess.run(
            cmd,
            capture_output=True,
            text=True,
            timeout=120,
            check=True,
        )
        self.container_id = result.stdout.strip()
        self.container_name = container_name
        self._container = self._client.containers.get(container_name)
        print(f"Started container {container_name} with ID {result.stdout.strip()}")

    def cleanup(self):
        if self.container_id:
            cmd = f"(timeout 60 docker stop {self.container_id} || docker rm -f {self.container_id}) >/dev/null 2>&1 &"
            subprocess.Popen(cmd, shell=True)
            print(f"Cleaned container {self.container_name} with ID {self.container_id}")

    def __del__(self):
        self.cleanup()

    def execute(self, command: str, workdir: str = "", timeout=180) -> dict[str, str]:
        workdir = workdir or self.workdir
        cmd = ["docker", "exec"]
        if workdir:
            cmd.extend(["-w", workdir])

        # Set environment variables
        # cmd.extend(["-e", f"{key}={value}"])

        cmd.extend([self.container_id, "bash", "-lc", command])

        result = super().execute(" ".join(cmd), workdir=workdir, timeout=timeout)
        return result
