from pathlib import Path
from typing import Any

from pydantic import BaseModel, ConfigDict

from lita.core.utils import apply_diff, read_file, replace_str_content


# Follow: https://docs.anthropic.com/en/docs/agents-and-tools/tool-use/implement-tool-use#example-of-a-good-tool-description
# NOTE: must implement a `Finish` tool so that we can stop interaction
class Tool(BaseModel):
    name: str
    description: str
    parameters: dict[str, Any] | None = None

    model_config = ConfigDict(arbitrary_types_allowed=True)

    def __call__(self, **kwargs) -> Any:
        return self.execute(**kwargs)

    def execute(self, **kwargs) -> str:
        raise NotImplementedError("Tool subclasses must implement execute method")

    async def aexecute(self, **kwargs) -> str:
        raise NotImplementedError("Tool subclasses must implement aexecute method")

    def schema(self) -> dict[str, Any]:
        return {
            "type": "function",
            "function": {
                "name": self.name,
                "description": self.description,
                "parameters": self.parameters,
            },
        }


# ---------------------------------------------
#        Common tools among all agents
# ---------------------------------------------


class BaseEditor(Tool):
    name: str = "base_editor"
    description: str = "Base editor for text files."
    parameters: dict = {
        "type": "object",
        "properties": {
            "operation": {
                "description": "File operation to perform.",
                "type": "string",
                "enum": ["view", "create", "edit"],
            },
        },
        "required": ["operation", "path"],
    }

    def view(self, path: str, view_range: list[int] = None, line_number: bool = False) -> str:
        _path = Path(path)
        if not _path.exists():
            return f"{self.name}: File not found: {path}"
        if view_range and (len(view_range) != 2 or not all(isinstance(i, int) for i in view_range)):
            return f"{self.name}: Invalid `view_range`: {view_range}."
        try:
            lines = read_file(_path, splitlines=True)
            if not lines:
                return ""

            if line_number:
                # max_line_num = len(lines)
                # width = len(str(max_line_num))
                lines = [f"{i}: {line}" for i, line in enumerate(lines, start=1)]

            if not view_range:
                return "".join(lines)  # Content should be the same as directly read
            else:
                start, end = view_range
                start = int(start)
                end = int(end)
                if start < 1 or start > end:
                    return f"{self.name}: Invalid `view_range`: {view_range}."
                return "".join(lines[start - 1 : end])
        except Exception as e:
            return f"{self.name}: Error reading file: {e}"

    def create(self, path: str, content: str) -> str:
        _path = Path(path)
        if _path.exists():
            return f"{self.name}: File already exists: {path}"
        try:
            _path.write_text(content, encoding="utf-8")
            return f"created: {path}"
        except Exception as e:
            return f"{self.name}: Error creating file: {e}"


class DiffBlockEditor(BaseEditor):
    name: str = "text_file_editor"
    description: str = """
TODO: fill this block
""".strip()  # noqa: E501
    parameters: dict = {
        "type": "object",
        "properties": {
            "operation": {
                "description": "File operation to perform.",
                "type": "string",
                "enum": ["view", "create", "edit"],
            },
        },
        "required": ["operation", "path"],
    }

    def edit(self, path: str, diff: str) -> str:
        _path = Path(path)
        if not _path.exists():
            return f"{self.name}: File not found: {path}"
        try:
            file_content = read_file(_path)
            stat, updated = apply_diff(file_content, diff)
            if stat:
                _path.write_text(updated)
                return "Edited the file."
            else:
                # Return the error message from apply_diff
                return updated
        except Exception as e:
            return f"{self.name}: Error editing file: {e}"


class StrReplaceEditor(BaseEditor):
    name: str = "str_replace_editor"
    description: str = """
TODO: fill this block
""".strip()  # noqa: E501
    parameters: dict = {
        "type": "object",
        "properties": {
            "operation": {
                "description": "File operation to perform.",
                "type": "string",
                "enum": ["view", "create", "edit"],
            },
        },
        "required": ["operation", "path"],
    }

    def edit(self, path: str, old_str: str, new_str: str) -> str:
        _path = Path(path)
        if not _path.exists():
            return f"{self.name}: File not found: {path}"
        try:
            file_content = read_file(_path)
            stat, updated = replace_str_content(file_content, old_str, new_str)
            if stat:
                _path.write_text(updated)
                return "Edited the file."
            else:
                # Return the error message from apply_diff
                return updated
        except Exception as e:
            return f"{self.name}: Error editing file: {e}"
