import json
import re
from pathlib import Path

from gitignore_parser import parse_gitignore
from rich.console import Console
from rich.panel import Panel
from rich.syntax import Syntax
from rich.table import Table


class PrettyPrinter:
    def __init__(self):
        self.console = Console()

    def print_message(self, role: str, content: str):
        """Print a formatted message with role-based styling"""
        role_styles = {
            "assistant": ("🤖", "cyan"),
            "user": ("👤", "green"),
            "tool output": ("🔧", "yellow"),
            "system": ("⚙️", "magenta"),
        }

        emoji, color = role_styles.get(role.lower(), ("📝", "white"))

        title = f"{emoji} {role.upper()}"
        panel = Panel(content, title=title, border_style=color, padding=(0, 1))
        self.console.print(panel, markup=False)

    def print_tool_call(self, tool_call):
        """Print a formatted tool call"""
        table = Table(show_header=False, box=None, padding=(0, 1))
        table.add_column("Key", style="bold blue")
        table.add_column("Value", style="white")

        table.add_row("🔧 Tool", tool_call.function.name)

        # Pretty print JSON arguments
        try:
            args = json.loads(tool_call.function.arguments)
            args_str = json.dumps(args, indent=2)
            syntax = Syntax(args_str, "json", theme="monokai", line_numbers=False)
            table.add_row("📋 Args", syntax)
        except Exception:
            table.add_row("📋 Args", tool_call.function.arguments)

        panel = Panel(table, title="🚀 Tool Call", border_style="blue", padding=(0, 1))
        self.console.print(panel)

    def print_separator(self):
        """Print a visual separator"""
        self.console.print("─" * 60, style="dim")

    def prompt_continue(self):
        """Pretty prompt for continuation"""
        return self.console.input("[bold yellow]Press Enter to continue...[/bold yellow] ")


ignored_dirs = {".git", ".venv", "__pycache__"}
ignored_files = set()


def apply_diff(content: str, diff: str) -> tuple[bool, str]:
    pattern = r"<<<<<<< SEARCH\n(.*?)=======\n(.*?)>>>>>>> REPLACE"
    mismatch_info = (
        f"Apply diff failed! No valid diff blocks found matching the pattern r\"{pattern}\"."
        + " Please check your diff format."
    )

    blocks = []
    matches = re.findall(pattern, diff, re.DOTALL)
    # If no matches found
    if len(matches) == 0:
        return (
            False,
            mismatch_info,
        )

    for s, r in matches:
        blocks.append((s, r))
    for old_str, new_str in blocks:
        new_content = content.replace(old_str, new_str)
        if new_content == content:
            if old_str != new_str:
                return (
                    False,
                    "Apply diff failed! "
                    + f"The string you want to replace: `{old_str}` does not exist in the original file. "
                    + "Please check content in your diff block.",
                )
            else:
                # We will ignore if the diff block does not contain actual changes (old = new)
                pass

        content = new_content

    return True, content


def replace_str_content(content: str, old_str: str, new_str: str) -> tuple[bool, str]:
    """
    One and only one `old_str` should be matched. Otherwise, an error will be raised.
    """
    num_appearance = content.count(old_str)
    if num_appearance != 1:
        return False, f"Expected exactly one occurrence of `{old_str}`, but found {num_appearance}."

    return True, content.replace(old_str, new_str)


def read_file(path: Path | str, splitlines: bool = False) -> str | list[str]:
    """
    \\n and \\r will be included in the returned content faithfully.
    """
    if isinstance(path, str):
        path = Path(path)
    if not path.exists():
        return "File not found: {path}"
    if path in ignored_files:
        return f"Reading {path} is not allowed."
    if splitlines:
        # NOTE: str.splitlines() is unsafe
        with open(path, encoding="utf-8") as f:
            return f.readlines()

    return path.read_text(encoding="utf-8")


def search_files(path: Path | str, query: str, file_type: str | None = None) -> list[Path]:
    # list all of files in the directory path
    # ignore the files in .gitignore
    # ignore the files by file_type
    if isinstance(path, str):
        path = Path(path)
    files = list(path.rglob("*"))

    files = [f for f in files if f not in ignored_files and f.is_file()]
    # filter ignored directories
    files = [f for f in files if not any(ignored in f.parts for ignored in ignored_dirs)]

    if file_type:
        files = [f for f in files if f.suffix == file_type]

    # TODO support .gitignore
    gitignore_path = path / ".gitignore"
    if gitignore_path.exists():
        gitignore = parse_gitignore(gitignore_path)
        files = [f for f in files if not gitignore(str(f))]

    hits = []
    for file in files:
        try:
            if file.is_file():
                file_content = file.read_text(encoding="utf-8", errors="ignore")
                if query in file_content:
                    hits.append(file)
        except Exception:
            continue
    return hits
