import os

import click

from lita.core.agent import LiteAgent, LiteAgentSR, LiteAgentTerminalOnly, LiteAgentWithSummarize
from lita.core.terminal.terminal_local import LocalTerminal


def generate_session_id() -> str:
    import datetime

    current_datetime = datetime.datetime.now()
    date_string = current_datetime.strftime("%Y%m%d%H%M%S")
    return date_string


@click.group()
@click.version_option(version="0.1.0")
def cli():
    """Lite Agent (lita) CLI"""
    pass


@cli.command()
@click.argument("task")
@click.option("--agent-type", "-a", help="Specific agent to use", default="lita")
@click.option("--model", "-m", help="Specific model to use", default="")
@click.option("--verbose", "-v", is_flag=True, help="Enable verbose output")
@click.option("--debug", "-d", is_flag=True, help="Enable debug mode")
@click.option("--save-tools-schema", "-s", is_flag=True, help="Save tools schema")
@click.option(
    "--traj_dir",
    "-o",
    help="Specific output directory for trajectories",
    default=".traj",
)
@click.option("--max_num_turns", "-n", type=int, default=50, help="Maximum number of turns for the agent")
@click.option("--timeout", "-t", type=int, default=180, help="Timeout for an LLM request in seconds")
def run(
    task: str,
    agent_type: str = "lita",
    model: str | None = None,
    verbose: bool = False,
    debug: bool = False,
    save_tools_schema: bool = False,
    traj_dir: str = ".traj",
    max_num_turns: int = 50,
    timeout: int = 180,
    max_retries: int = 3,
):
    # if task is a file, read the content as the task
    if os.path.isfile(task.strip()):
        with open(task) as f:
            task = f.read().strip()
    run_id = generate_session_id()
    python_env = os.getenv("PYTHON_ENV", None)
    terminal = LocalTerminal(python_env=python_env)

    agent_cls = LiteAgent
    if agent_type == "lita-mini":
        agent_cls = LiteAgentTerminalOnly
    elif agent_type == "lita-summarize":
        agent_cls = LiteAgentWithSummarize
    elif agent_type == "lita-sr":
        agent_cls = LiteAgentSR

    agent = agent_cls(
        run_id=run_id,
        model=model,
        verbose=verbose,
        debug=debug,
        max_num_turns=max_num_turns,
        timeout=timeout,
        max_retries=max_retries,
        terminal=terminal,
    )
    agent.run(task)
    agent.save(save_dir=traj_dir)
    if save_tools_schema:
        tools_schema_file = f"{traj_dir}/{run_id}_tools_schema.json"
        agent.save_tools_schema(save_file=tools_schema_file)


def main():
    cli()


if __name__ == "__main__":
    main()
