import gc
import json
import resource
import sys
from pathlib import Path

from lita.core.memory import Memory


def read_jsonl(file_path: Path | str) -> list[dict]:
    if isinstance(file_path, str):
        file_path = Path(file_path)

    if not file_path.exists():
        return []

    with open(file_path, encoding="utf-8") as f:
        return [json.loads(line) for line in f if line.strip()]


def write_jsonl(file_path: Path | str, data: list[dict]):
    if isinstance(file_path, str):
        file_path = Path(file_path)

    with open(file_path, "w", encoding="utf-8") as f:
        for item in data:
            f.write(json.dumps(item) + "\n")


def set_memory_limits(max_memory_gb=16):
    try:
        max_memory_bytes = max_memory_gb * 1024 * 1024 * 1024

        # Set VMM size
        resource.setrlimit(resource.RLIMIT_AS, (max_memory_bytes, max_memory_bytes))

        # Set data segment limit (heap memory)
        try:
            resource.setrlimit(resource.RLIMIT_DATA, (max_memory_bytes, max_memory_bytes))
        except (ValueError, OSError) as e:
            # noqa
            print(f"Set data segment limit not supported on this machine: {e}")

        # print(f"Memory limit set to {max_memory_gb}GB for process {os.getpid()}")
        return True

    except Exception as e:
        print(f"Warning: Could not set memory limit: {e}", file=sys.stderr)
        return False


def cleanup_resources():
    try:
        # Force gc
        gc.collect()

        # Clean litellm's cache (if exists)
        try:
            import litellm

            if hasattr(litellm, 'cache'):
                litellm.cache = None
        except ImportError:
            pass

    except Exception as e:
        print(f"Error during cleanup: {e}", file=sys.stderr)


def load_memory_from_jsonl(file_path: Path | str, limit: int = None) -> Memory:
    messages = read_jsonl(file_path)

    memory = Memory()

    if limit is None:
        limit = len(messages)

    for msg in messages[:limit]:
        memory.add_message(
            role=msg.get("role"),
            content=msg.get("content"),
            tool_calls=msg.get("tool_calls"),
            tool_call_id=msg.get("tool_call_id"),
        )

    return memory
