import argparse
import os
import torch
from exp.exp_main import Exp_Main
import random
import numpy as np

def set_seed(seed):
    random.seed(seed)
    seed += 1
    np.random.seed(seed)
    seed += 1
    torch.manual_seed(seed)

parser = argparse.ArgumentParser(description='LGPred generating linear predictor from decomposed time series')

# basic config
parser.add_argument('--model_id', type=str, required=False, default='ECL', help='model id')
parser.add_argument('--model', type=str, required=False, default='LGPred',
                    help='model name')

# data loader
parser.add_argument('--data', type=str, required=False, default='custom', help='dataset type')
parser.add_argument('--root_path', type=str, default='./dataset/ETT-small', help='root path of the data file')
parser.add_argument('--data_path', type=str, default='ETTh1.csv', help='data file')
parser.add_argument('--features', type=str, default='M',
                    help='forecasting task, options:[M, S, MS]; M:multivariate predict multivariate, S:univariate predict univariate, MS:multivariate predict univariate')
parser.add_argument('--target', type=str, default='OT', help='target feature in S or MS task')
parser.add_argument('--freq', type=str, default='h',
                    help='freq for time features encoding, options:[s:secondly, t:minutely, h:hourly, d:daily, b:business days, w:weekly, m:monthly], you can also use more detailed freq like 15min or 3h')
parser.add_argument('--checkpoints', type=str, default='./checkpoints/', help='location of model checkpoints')

# forecasting task
parser.add_argument('--seq_len', type=int, required=False, default=336, help='input sequence length')
parser.add_argument('--pred_len', type=int, required=False,  default=96, help='prediction sequence length')
parser.add_argument('--label_len', type=int, default=0, help='start token length')

# model define
parser.add_argument('--enc_in', type=int, default=321, help='encoder input size')
parser.add_argument('--d_feat', type=int, default=512, help='dimension of attention')
parser.add_argument('--d_rep', type=int, default=256, help='dimension of conv block')
parser.add_argument('--d_latent', type=int, default=256, help='dimension of conv block')
parser.add_argument('--kernel_size', type=int, default=3, help='kernel size of conv block')
parser.add_argument('--n_layer', type=int, default=1, help='num of layers')
parser.add_argument('--n_heads', type=int, default=4, help='num of heads')
parser.add_argument('--dropout', type=float, default=0, help='dropout')
parser.add_argument('--embed', type=str, default='timeF',
                    help='time features encoding, options:[timeF, fixed, learned]')
parser.add_argument('--min_lr', type=float, default=1e-30)
parser.add_argument('--warmup_epochs', type=int, default=3)
parser.add_argument('--std', type=float, default=0.2)

# optimization
parser.add_argument('--optim', type=str, default='adam', help='optimizer')
parser.add_argument('--num_workers', type=int, default=10, help='data loader num workers')
parser.add_argument('--itr', type=int, default=1, help='experiments times')
parser.add_argument('--train_epochs', type=int, default=15, help='train epochs')
parser.add_argument('--batch_size', type=int, default=16, help='batch size of train input data')
parser.add_argument('--patience', type=int, default=5, help='early stopping patience')
parser.add_argument('--learning_rate', type=float, default=3e-4, help='optimizer learning rate')
parser.add_argument('--des', type=str, default='Exp', help='exp description')
parser.add_argument('--lradj', type=str, default='exponential_with_warmup', help='adjust learning rate')
parser.add_argument('--weight_decay', type=float, default=0, help='weight decay rate')

# GPU
parser.add_argument('--use_gpu', type=bool, default=True, help='use gpu')
parser.add_argument('--gpu', type=int, default=0, help='gpu')
parser.add_argument('--use_multi_gpu', action='store_true', help='use multiple gpus', default=False)
parser.add_argument('--devices', type=str, default='0,1,2,3', help='device ids of multile gpus')
# Trainer option 
parser.add_argument('--verbose', action='store_true', help='control print option', default=False)
parser.add_argument('--retrain', action='store_true', help='control retraining option', default=False)

args = parser.parse_args()

args.use_gpu = True if torch.cuda.is_available() and args.use_gpu else False

if args.use_gpu and args.use_multi_gpu:
    args.dvices = args.devices.replace(' ', '')
    device_ids = args.devices.split(',')
    args.device_ids = [int(id_) for id_ in device_ids]
    args.gpu = args.device_ids[0]

print('Args in experiment:')
print(args)

Exp = Exp_Main
torch.set_num_threads(2)

for ii in range(args.itr):
    set_seed(ii)
    # setting record of experiments
    setting = '{}_{}_{}_ft{}_sl{}_pl{}_dc{}_dd{}_ks{}_df{}_nl{}_lr{}_do{}_{}_{}'.format(
        args.model_id,
        args.model,
        args.data,
        args.features,
        args.seq_len, 
        args.pred_len, 
        args.d_rep, 
        args.d_latent,
        args.kernel_size, 
        args.d_feat,
        args.n_layer, 
        args.learning_rate,
        args.dropout,
        args.des, ii)
    if not args.retrain:
        if os.path.exists(os.path.join('./results/', setting)):
            continue

    exp = Exp(args)  # set experiments
    print('>>>>>>>start training : {}>>>>>>>>>>>>>>>>>>>>>>>>>>'.format(setting))
    exp.train(setting)

    print('>>>>>>>testing : {}<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<'.format(setting))
    exp.test(setting, data='val')
    exp.test(setting, data='test')

    torch.cuda.empty_cache()
