import json


def load_json(file_path):
   with open(file_path, 'r', encoding='utf-8') as file:
        data = json.load(file)
   return data


def generate_motion(self, measuremeng_path):
    """
    Rule-based method to generate the motion of the current frame based on the measurement.json file from Carla.
    We separate the instruction into two parts: ## perception ## and ## motion ##.
    We put the perception part in the front of the motion to let perception guide the motion.

    Args:
        measurement_path (str): The path of measurement.json file.

    Returns:
        instruction (str): The instruction for the current frame.
    """

    data = load_json(measuremeng_path)
    ## perception ##
    perception_instructions = []
    
    # juntion
    if data['is_junction']:
        perception_instructions.append("Approaching a junction, prepare to follow traffic rules.")
    
    # junction vehicles
    if len(data["is_junction_vehicle_present"]) == 1:
        perception_instructions.append("A vehicle is present at the junction. Be cautious.")
    elif len(data["is_junction_vehicle_present"]) > 1:
        perception_instructions.append("Multiple vehicles are present at the junction. Be cautious.")

    # vehicles
    if len(data["is_vehicle_present"]) == 1:
        perception_instructions.append("Watch out for the car ahead, there's a vehicle in front.")
    elif len(data["is_vehicle_present"]) > 1:
        perception_instructions.append("Watch out for the cars ahead, there are multiple vehicles in front.")

    # lane vehicles
    if len(data["is_lane_vehicle_present"]) == 1:
        perception_instructions.append("A vehicle is present in the lane. Be cautious.")
    elif len(data["is_lane_vehicle_present"]) > 1:
        perception_instructions.append("Multiple vehicles are present in the lane. Be cautious.")

    # bikes
    if len(data["is_bike_present"]) == 1:
        perception_instructions.append("There is a bike ahead. Be cautious.")
    elif len(data["is_bike_present"]) > 1:
        perception_instructions.append("Multiple bikes are ahead. Be cautious.")

    # pedestrians
    if len(data["is_pedestrian_present"]) == 1:
        perception_instructions.append("There is a pedestrian ahead. Be cautious.")
    elif len(data["is_pedestrian_present"]) > 1:
        perception_instructions.append("Multiple pedestrians are ahead. Be cautious.")

    # red light
    if len(data["is_red_light_present"]) > 0:
        perception_instructions.append("There is a red light ahead.")
    # stop sign
    if len(data['is_stop_sign_present']) > 0:
        perception_instructions.append("There is a stop sign ahead.")
    
    ## motion ##
    motion_instructions = []
    # speed
    if data["should_slow"]:
        motion_instructions.append("Slow down to ensure safety.")
    else:
        if data['speed'] < data['target_speed']:
            speed_difference = data['target_speed'] - data['speed']
            if data['speed'] == 0.0:
                if not data['brake']:
                    motion_instructions.append("Start accelerating gradually towards the target speed.")
                else:
                    motion_instructions.append("Remain stopped due to brake application.")
            elif speed_difference > 5.0:
                motion_instructions.append("Significantly below target speed, accelerate if safe.")
            else:
                motion_instructions.append("Slightly below target speed, gently increase acceleration.")
        elif data['speed'] > data['target_speed']:
            motion_instructions.append("Above target speed, decelerate.")
        else:
            motion_instructions.append("Maintain current speed to match the target speed.")

    # steer
    # when steer is less than 0.1, we consider it as straight.
    if data['steer'] > 0.1:
        if data['steer'] > 0.5:
            motion_instructions.append("Steer right sharply.")
        else:
            motion_instructions.append("Make a slight right turn.")
    elif data['steer'] < -0.1:
        if data['steer'] < -0.5:
            motion_instructions.append("Steer left sharply.")
        else:
            motion_instructions.append("Make a slight left turn.")
    else:
        motion_instructions.append("Keep the steering wheel straight.")

    if data["brake"] or data["should_brake"]:
        motion_instructions = [] # if should brake, just brake.
        motion_instructions.append("Apply brakes safely.")
    
    perception = " ".join(perception_instructions)
    motion = " ".join(motion_instructions)
    
    return " ".join([perception, motion])




    



            




