import os
from typing import Any, Dict

from tap import Tap

from src import config
from src.utils import helpers


class ArgumentsBase(Tap):
    # score
    score_name: str
    score_kwargs: Dict[str, Any] = {}

    # experiment
    _output_dir: str = None
    debug: bool = False

    # model
    model_name: str = "densenet121_cifar10"
    chkpt_path: str = None

    # datamodule
    dataset_name: str = None
    data_dir: str = config.DATASETS_DIR
    num_workers: int = 2
    val_split: float = 0
    batch_size: int = 64
    seed: int = 42
    pin_memory: bool = True

    def configure(self) -> None:
        # this is ran first
        self.add_argument("--score_kwargs", type=helpers.str_to_dict)

    def process_args(self):
        os.makedirs(self.output_dir, exist_ok=True)
        if self.dataset_name is None:
            self.dataset_name = self.model_name.split("_")[1].lower()

    @property
    def output_dir(self):
        if self._output_dir is None:
            path = os.path.join(
                config.RESULTS_DIR,
                self.model_name,
                str(self.seed),
            )
            if self.val_split > 0:
                path = os.path.join(path, f"val_split_{self.val_split}")
            path = os.path.join(path, self.score_name)
            return path
        return self._output_dir

    @property
    def url(self) -> str:
        if self.chkpt_path is None:
            return f"{self.seed}/best_weights-v0.pth"
        return self.chkpt_path

    @property
    def results_filename(self) -> str:
        return os.path.join(config.RESULTS_DIR, "results.csv")
