import json
import re


def convert_to_pddl(lines):
    actions = []
    for line in lines:
        line = line.strip().lower()  # Normalize case and strip whitespace

        # Match and convert each action
        if "unstack" in line:
            match = re.search(r"unstack.*block (\w).*block (\w)", line)
            if match:
                actions.append(
                    f"(unstack {match.group(1).lower()} {match.group(2).lower()})"
                )
        elif "put down" in line:
            match = re.search(r"put down.*block (\w)", line)
            if match:
                actions.append(f"(put-down {match.group(1).lower()})")
        elif "pick up" in line:
            match = re.search(r"pick up.*block (\w)", line)
            if match:
                actions.append(f"(pick-up {match.group(1).lower()})")
        elif "stack" in line:
            match = re.search(r"stack.*block (\w).*block (\w)", line)
            if match:
                actions.append(
                    f"(stack {match.group(1).lower()} {match.group(2).lower()})"
                )

    return "\n".join(actions)


def parse_to_val(answer):
    actions = answer.split("\n")
    actions = convert_to_pddl(actions)
    return actions


def remove_putdown_pickup_pairs(commands):
    # Split the input into individual lines
    lines = commands.strip().splitlines()

    # Iterate through lines and keep track of those to keep
    filtered_lines = []
    skip_next = False

    for i in range(len(lines) - 1):
        if skip_next:
            skip_next = False
            continue

        current_line = lines[i].strip()
        next_line = lines[i + 1].strip()

        # Check for the (put-down x) followed by (pick-up x) pattern
        if current_line.startswith("(put-down") and next_line.startswith("(pick-up"):
            x = current_line.split()[1]
            y = next_line.split()[1]
            if x == y:
                skip_next = True  # Skip the next line
            else:
                filtered_lines.append(current_line)
        else:
            filtered_lines.append(current_line)

    # Add the last line if it wasn't part of a (put-down x) followed by (pick-up x) pattern
    if not skip_next:
        filtered_lines.append(lines[-1].strip())

    # Return the filtered list as a single string
    return "\n".join(filtered_lines)


def parse_problem(input_str):
    input_str = "\n".join([line for line in input_str.splitlines() if line.strip()])
    return parse_init_section(input_str), parse_goal_section(input_str)


def parse_init_section(problem_str):
    # print(f"problem_str: {problem_str}")

    # Split the string into lines
    lines = problem_str.split("\n")

    # find the elements between the (:init and the first )
    init_section = lines[lines.index("(:init") + 1 : lines.index(")")]

    structure = []

    for line in init_section:
        # if line starts with (ontable
        if line.startswith("(ontable"):
            structure.append(line[-2])

    for _ in range(10):
        for i in range(len(structure)):
            block = structure[i][0]
            for line in init_section:
                if line.startswith("(on ") and line.endswith(f" {block})"):
                    structure[i] = (
                        "Block "
                        + line[4][0].upper()
                        + line[4][1:]
                        + " on "
                        + "Block "
                        + structure[i].upper()
                    )

    for i in range(len(structure)):
        structure[i] = structure[i] + " on table"

    return structure


def chain_relations(goal_section):
    # A dictionary to store the blocks on top of others
    stack_dict = {}

    for item in goal_section:
        blocks = item.strip("()").split()
        if blocks[2] not in stack_dict:
            stack_dict[blocks[2]] = blocks[1]
        else:
            # Find the topmost block and add the new block on top
            current = blocks[2]
            while current in stack_dict:
                current = stack_dict[current]
            stack_dict[current] = blocks[1]

    # Construct the output by identifying bases that aren't on top of any other block
    all_blocks = set(stack_dict.keys()).union(set(stack_dict.values()))
    base_blocks = all_blocks.difference(set(stack_dict.values()))

    formatted_output = []
    for base in base_blocks:
        # Start from the base and go up
        stack = [base.upper()]
        current = base.lower()
        while current in stack_dict:
            next_block = stack_dict[current].upper()
            stack.append(next_block)
            current = next_block.lower()
        formatted_output.append(
            " on ".join(stack[::-1])
        )  # Reverse the stack to get top to bottom

    return ", ".join(formatted_output)


def parse_goal_section(problem_str):
    # Split the string into lines
    lines = problem_str.split("\n")
    # find the elements between the (:init and the first )
    goal_section = lines[lines.index("(:goal") + 2 : -2]
    goal_section[-1] = goal_section[-1][:-1]

    # print(f"goal_section: {goal_section}")

    relations = chain_relations(goal_section)
    # print(f"relations: {relations}")

    return relations


def extract_ground_truths_from_file(filename):
    # Open and read the JSON data from the file
    with open(filename, "r") as file:
        data = json.load(file)

    # Extract instances
    instances = data.get("instances", [])

    # Create a dictionary where the key is the instance ID as a string and the value is the ground truth plan
    ground_truths_dict = {
        str(instance["instance_id"]): "\n".join(instance["ground_truth_plan"])
        for instance in instances
    }

    return ground_truths_dict
